//
//  FCIMInputController.m
//  FCIM
//
//  Created by Andrew Choi on 22/08/08.
//  Copyright 2008 Andrew Choi. All rights reserved.
//

/*
 
 Permission for the use of this code is granted only for research, educational, and non-commercial purposes.
 
 Redistribution of this code or its parts in source, binary, and any other form without permission, with or without modification, is prohibited.  Modifications include, but are not limited to, translation to other programming languages and reuse of tables, constant definitions, and API's defined in it.
 
 Andrew Choi is not liable for any losses or damages caused by the use of this software.
 
 */

#import "FCIMInputController.h"

#import "FCIMCandidates.h"
#import "ConversionEngine.h"

@implementation FCIMInputController

- (id)initWithServer:(IMKServer*)server delegate:(id)delegate client:(id)inputClient
{
    self = [super initWithServer:server delegate:delegate client:inputClient];
    
	if (self)
	{
		client = inputClient;
		compositionBuffer = [[NSMutableString alloc] initWithString:@""];
		currentCandidates = nil;
    }
    
	return self;
}

- (void)dealloc
{
	[compositionBuffer release];
	
	[super dealloc];
}

- (void)findCurrentCandidates
{
	currentCandidates = [[[ConversionEngine sharedInstance] characters] objectForKey:compositionBuffer];
	
	if (!currentCandidates)
		currentCandidates = [[[ConversionEngine sharedInstance] phrases] objectForKey:compositionBuffer];
	
	if (currentCandidates)
		[currentCandidates retain];
}

- (void)updateCandidatesPanel
{
	if (currentCandidates)
	{
		// Need to set font here; setting it in init... doesn't work.		
		NSUserDefaults *defaultsDictionary = [NSUserDefaults standardUserDefaults];

		NSString *candidateFontName = [defaultsDictionary objectForKey:@"candidateFontName"];
		float candidateFontSize = [[defaultsDictionary objectForKey:@"candidateFontSize"] floatValue];
		
		NSFont *candidateFont = [NSFont fontWithName:candidateFontName size:candidateFontSize];
		[[FCIMCandidates sharedInstance] setAttributes:[NSDictionary dictionaryWithObject:candidateFont forKey:NSFontAttributeName]];
		
		[[FCIMCandidates sharedInstance] setPanelType:[defaultsDictionary integerForKey:@"candidatePanelType"]];		
		
		[[FCIMCandidates sharedInstance] updateCandidates];
		[[FCIMCandidates sharedInstance] show:kIMKLocateCandidatesBelowHint];
	}
	else
		[[FCIMCandidates sharedInstance] hide];
}

- (NSArray*)candidates:(id)sender
{
	return currentCandidates;	
}

- (void)candidateSelectionChanged:(NSAttributedString*)candidateString
{
	// Intentionally blank.
}

- (void)clearCompositionBuffer
{
	[compositionBuffer deleteCharactersInRange:NSMakeRange(0, [compositionBuffer length])];	
}

- (void)candidateSelected:(NSAttributedString*)candidateString
{
	[client insertText:candidateString replacementRange:NSMakeRange(NSNotFound, 0)];
	
	[self clearCompositionBuffer];
	
	[currentCandidates release];
	currentCandidates = nil;
}

- (void)commitComposition:(id)sender
{
	[sender insertText:compositionBuffer replacementRange:NSMakeRange(NSNotFound, 0)];
	
	[self clearCompositionBuffer];
	
	[currentCandidates release];
	currentCandidates = nil;
}

- (id)composedString:(id)sender
{
	return [[[NSAttributedString alloc] initWithString:compositionBuffer] autorelease];
}

- (NSString *)convertPunctuation:(NSString*)string
{
	NSDictionary *punctuations = [[ConversionEngine sharedInstance] punctuations];
	
	return [punctuations objectForKey:[compositionBuffer stringByAppendingString:string]];
}

- (BOOL)areValidCharacters:(NSString*)string
{
	NSCharacterSet *validCharacterSet = [NSCharacterSet lowercaseLetterCharacterSet];
	
	NSScanner *scanner = [NSScanner scannerWithString:string];
	
	return [scanner scanCharactersFromSet:validCharacterSet intoString:NULL] && [scanner isAtEnd];
}

- (BOOL)inputText:(NSString*)string client:(id)sender
{
	//Return YES to indicate the the key input was received and dealt with.
	//Key processing will not continue in that case.  In
	//other words the system will not deliver a key down event to the application.
	//Returning NO means the original key down will be passed on to the client.
	
	NSString *punctuation = [self convertPunctuation:string];
	if (punctuation)
	{
		[client insertText:punctuation replacementRange:NSMakeRange(NSNotFound, 0)];
		
		[self clearCompositionBuffer];
		
		return YES;
	}
	else if ([string isEqualToString:@" "])
	{
		if ([compositionBuffer length] == 0)
		{
			[client insertText:@" " replacementRange:NSMakeRange(NSNotFound, 0)];
		}
		else
		{
			if (currentCandidates)
			{
				[self candidateSelected:[currentCandidates objectAtIndex:0]];
			}
			else
			{
				NSBeep();
			}
		}
		
		return YES;
	}
	else if ([self areValidCharacters:string])
	{
		[compositionBuffer appendString:string];
		
		[self findCurrentCandidates];

		[self updateComposition];

		[self updateCandidatesPanel];
	
		return YES;
	}
	else
		return NO;
}

- (void)deleteBackward:(id)sender
{
	// We're called only when [compositionBuffer length] > 0
	[compositionBuffer deleteCharactersInRange:NSMakeRange([compositionBuffer length] - 1, 1)];

	[self findCurrentCandidates];

	[self updateComposition];

	[self updateCandidatesPanel];
}

-(BOOL)didCommandBySelector:(SEL)aSelector client:(id)sender
{
    if ([self respondsToSelector:aSelector] && [compositionBuffer length] > 0)
	{
		[self performSelector:aSelector withObject:sender];
		return YES; 
	}
	else
		return NO;
}

- (NSMenu*)menu
{
	return [[NSApp delegate] menu];
}

@end
