//
//  Device.m
//  MIDIDevicePopupTest
//
//  Created by Andrew Choi on 16/08/05.
//  Copyright 2005 Andrew Choi.  Perl Artistic License.

#import "AliasUtilities.h"
#import "MIDINameDocumentUtilities.h"

#import "PatchBank.h"

#import "Device.h"

@implementation Device

- (id)initWithDeviceRef:(MIDIDeviceRef)newDeviceRef
{
  if (self = [super init])
  {
    deviceRef = newDeviceRef;

    isMasterNameDocumentLoaded = NO;
    
    deviceModes = nil;
  }
  return self;
}

- (void)delloc
{
  [deviceModes release];
  
  [super dealloc];
}

- (NSString *)name
{
  CFStringRef cfName;
  MIDIObjectGetStringProperty(deviceRef, kMIDIPropertyName, &cfName);

  return [(NSString *)cfName autorelease];
}

- (NSDictionary *)nameConfiguration
{
  CFDictionaryRef cfNameConfiguration;
  if (MIDIObjectGetDictionaryProperty(deviceRef, kMIDIPropertyNameConfiguration, &cfNameConfiguration) != noErr)
    return nil;  /* Call fails if property has not been set.  */
  else
    return [(NSDictionary *)cfNameConfiguration autorelease];
}

- (void)setNameConfiguration:(NSDictionary *)newNameConfiguration
{
  if (MIDIObjectSetDictionaryProperty(deviceRef, kMIDIPropertyNameConfiguration, (CFDictionaryRef) newNameConfiguration) != noErr)
    RAISE_EXCEPTION(@"Fail to set Name Configuration Dictionary");
}

- (void)clearNameConfiguration
{
  deviceModes = nil;
  isMasterNameDocumentLoaded = NO;
  
  if (MIDIObjectRemoveProperty(deviceRef, kMIDIPropertyNameConfiguration) != noErr)
    RAISE_EXCEPTION(@"Fail to remove Name Configuration Dictionary");
}

- (NSString *)masterNameDocument
{
  NSDictionary *nameConfiguration = [self nameConfiguration];
  
  if (!nameConfiguration)
    return nil;
  
  CFDataRef masterAliasData = (CFDataRef)[nameConfiguration objectForKey:@"master"];
  
  if (!masterAliasData)
    RAISE_EXCEPTION(@"Master name document path missing from name config");
  
  return POSIXPathnameFromAliasData(masterAliasData);
}

- (NSString *)masterNameDocumentDisplayName
{
  return [[NSFileManager defaultManager] displayNameAtPath:[self masterNameDocument]];
}

- (void)setMasterNameDocument:(NSString *)newMasterNameDocument
{
  CFDataRef masterAliasData = AliasDataFromPOSIXPathname(newMasterNameDocument);

  NSMutableDictionary *newNameConfiguration = [NSMutableDictionary dictionaryWithDictionary: [self nameConfiguration]];
  
  [newNameConfiguration setObject:(NSData *)masterAliasData forKey:@"master"];
  CFRelease(masterAliasData);
  
  NSXMLDocument *xmlDoc = XMLDocFromFile(newMasterNameDocument);  
  NSArray *myDeviceModes = DeviceModesFromXMLDocument(xmlDoc);
  [newNameConfiguration setObject:[[myDeviceModes objectAtIndex:0] objectForKey:@"name"] forKey:@"currentDeviceMode"];

  [newNameConfiguration setObject:CurrentChannelNameSetFromDeviceMode([myDeviceModes objectAtIndex:0]) forKey:@"currentChannelNameSets"];
  
  [self setNameConfiguration:newNameConfiguration];
}

- (IBAction)chooseMasterNameDocument:(id)sender
{
  NSOpenPanel *op = [NSOpenPanel openPanel];
  [op setAllowsMultipleSelection:NO];
  NSArray *fileTypes = [NSArray arrayWithObjects:@"midnam", @"xml", nil];
  int result;
  NSString *oldMasterNameDocument = [self masterNameDocument];
  if (oldMasterNameDocument)
    result = [op runModalForDirectory:[oldMasterNameDocument stringByDeletingLastPathComponent] file:[oldMasterNameDocument lastPathComponent] types:fileTypes];
  else
    result = [op runModalForDirectory:nil file:nil types:fileTypes];
  
  if (result == NSOKButton)
    [self setMasterNameDocument:[[op filenames] objectAtIndex:0]];
}

- (IBAction)clearMasterNameDocument:(id)sender
{
  [self clearNameConfiguration];
}

- (NSDictionary *)banks
{
  NSDictionary *nameConfiguration = [self nameConfiguration];
  
  if (!nameConfiguration)
    return nil;
  
  return (NSDictionary *)[nameConfiguration objectForKey:@"banks"];
}

- (void)loadMasterNameDocument
{
  if (!isMasterNameDocumentLoaded)
  {
    isMasterNameDocumentLoaded = YES;
    
    NSString *myMasterNameDocument = [self masterNameDocument];
    if (myMasterNameDocument)
    {
      NSXMLDocument *xmlDoc = XMLDocFromFile(myMasterNameDocument);
    
      deviceModes = [DeviceModesFromXMLDocument(xmlDoc) retain];
      
      patchBanks = [PatchBanksFromXMLDocument(xmlDoc, self, [self banks]) retain];
    }
  }
}

- (unsigned int)countOfDeviceModes
{
  [self loadMasterNameDocument];
  
  return [deviceModes count];
}

- (NSString *)objectInDeviceModesAtIndex:(unsigned int)index
{
  [self loadMasterNameDocument];
  
  return [deviceModes objectAtIndex:index];
}

- (NSDictionary *)currentDeviceMode
{
  NSString *myCurrentDeviceModeName = [[self nameConfiguration] objectForKey:@"currentDeviceMode"];
  
  if (!myCurrentDeviceModeName)
    return nil;
  
  [self loadMasterNameDocument];

  for (int i = 0; i < [deviceModes count]; i++)
    if ([myCurrentDeviceModeName isEqualToString:[[deviceModes objectAtIndex:i] objectForKey:@"name"]])
      return [deviceModes objectAtIndex:i];
  
  RAISE_EXCEPTION(@"Current device mode in nameconfig not found in master name document");

  return nil;
}

- (void)setCurrentDeviceMode:(NSDictionary *)newCurrentDeviceMode
{
  NSMutableDictionary *newNameConfiguration = [NSMutableDictionary dictionaryWithDictionary:[self nameConfiguration]];
  
  [newNameConfiguration setObject:[newCurrentDeviceMode objectForKey:@"name"] forKey:@"currentDeviceMode"];
  
  [newNameConfiguration setObject:CurrentChannelNameSetFromDeviceMode(newCurrentDeviceMode) forKey:@"currentChannelNameSets"];

  [self setNameConfiguration:newNameConfiguration];
}

- (unsigned int)countOfPatchBanks
{
  [self loadMasterNameDocument];
  
  return [patchBanks count];
}

- (PatchBank *)objectInPatchBanksAtIndex:(unsigned int)index
{
  [self loadMasterNameDocument];
  
  return [patchBanks objectAtIndex:index];
}

- (void)setPatchNameList:(NSString *)patchNameList andFile:(NSString *)file forPatchBank:(NSString *)patchBank
{
  NSMutableDictionary *newNameConfiguration = [NSMutableDictionary dictionaryWithDictionary:[self nameConfiguration]];
  
  NSMutableDictionary *newBanks = [NSMutableDictionary dictionaryWithDictionary:[self banks]];
  
  if ([file length] != 0)
    [newBanks setObject:[NSDictionary dictionaryWithObjectsAndKeys:patchNameList, @"patchNameList", AliasDataFromPOSIXPathname(file), @"file", nil] forKey:patchBank];
  else
    [newBanks removeObjectForKey:patchBank];
    
  [newNameConfiguration setObject:newBanks forKey:@"banks"];
  
  [self setNameConfiguration:newNameConfiguration];  
}

@end
