%include exception.i

%{
#include <CoreFoundation/CoreFoundation.h>

#include "exception.h"

static int C_word_is_alist(C_word c_word)
{
    while (c_word != C_SCHEME_END_OF_LIST)
    {
	if (!C_swig_is_pair(c_word) || !C_swig_is_pair(C_u_i_car(c_word)))
	    return 0;

	c_word = C_u_i_cdr(c_word);
    }

    return 1;
}

static int alist_length(C_word c_word)
{
    int len = 0;

    while (c_word != C_SCHEME_END_OF_LIST)
    {
	len ++;
	c_word = C_u_i_cdr(c_word);
    }

    return len;
}

CFPropertyListRef CFPropertyListFromCWord(C_word c_word)
{
    if (C_swig_is_string(c_word))
	/* Strings returned by C_c_string are not null-terminated when lengths are multiples of four! */
	return CFStringCreateWithBytes(NULL, (const UInt8 *) C_c_string(c_word), C_header_size(c_word), kCFStringEncodingUTF8, 0);
    else if (C_truep(C_blockp(c_word)) && C_truep(C_bytevectorp(c_word)))
    {
	return CFDataCreate(NULL, (const UInt8 *) C_c_bytevector(c_word), C_header_size(c_word));
    }
    else if (C_swig_is_fixnum(c_word))
    {
	long l = C_num_to_long(c_word);
	return CFNumberCreate(kCFAllocatorDefault, kCFNumberLongType, &l);
    }
    else if (C_swig_is_flonum(c_word))
    {
	double d = C_flonum_magnitude(c_word);
	return CFNumberCreate(kCFAllocatorDefault, kCFNumberDoubleType, &d);
    }
    else if (C_swig_is_vector(c_word))
    {
	int len = C_header_size(c_word);

	CFPropertyListRef *values = CFAllocatorAllocate(kCFAllocatorDefault, sizeof(CFPropertyListRef) * len, 0);
	if (!values)
	    return NULL;

	for (int i = 0; i < len; i++)
	{
	    C_word o = C_block_item(c_word, i);

	    values[i] = CFPropertyListFromCWord(o);
	    if (!values[i])
	    {
		CFAllocatorDeallocate(kCFAllocatorDefault, values);
		return NULL;
	    }
	}

	CFArrayRef a = CFArrayCreate(kCFAllocatorDefault, values, len, &kCFTypeArrayCallBacks);

	for (int i = 0; i < len; i++)
	    CFRelease(values[i]);
	CFAllocatorDeallocate(kCFAllocatorDefault, values);

	return a;
    }
    else if (C_word_is_alist(c_word))
    {
	int size = alist_length(c_word);

	CFPropertyListRef *keys = CFAllocatorAllocate(kCFAllocatorDefault, sizeof(CFPropertyListRef) * size, 0);
	CFPropertyListRef *values = CFAllocatorAllocate(kCFAllocatorDefault, sizeof(CFPropertyListRef) * size, 0);
	if (!keys || !values)
	    return NULL;

	int i = 0;
	C_word curr_elt = c_word;
	while (curr_elt != C_SCHEME_END_OF_LIST)
	{
	    C_word key = C_u_i_car(C_u_i_car(curr_elt));
	    C_word value = C_u_i_cdr(C_u_i_car(curr_elt));

	    keys[i] = CFPropertyListFromCWord(key);
	    values[i] = CFPropertyListFromCWord(value);

	    if (!keys[i] || !values[i])
	    {
		CFAllocatorDeallocate(kCFAllocatorDefault, keys);
		CFAllocatorDeallocate(kCFAllocatorDefault, values);
		return NULL;
	    }

	    i++;
	    curr_elt = C_u_i_cdr(curr_elt);
	}

	CFDictionaryRef d = CFDictionaryCreate(kCFAllocatorDefault, keys, values, size, &kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks);

	for (int i = 0; i < size; i++)
	{
	    CFRelease(keys[i]);
	    CFRelease(values[i]);
	}
	CFAllocatorDeallocate(kCFAllocatorDefault, keys);
	CFAllocatorDeallocate(kCFAllocatorDefault, values);

	return d;
    }
    else
	return NULL;
}

C_word CWordFromCFPropertyList(CFPropertyListRef plist);

static void add_value_key_pair(const void *key, const void *value, void *d)
{
    C_word c_key = CWordFromCFPropertyList((CFPropertyListRef) key);
    C_word c_value = CWordFromCFPropertyList((CFPropertyListRef) value);

    C_word *head = d;

    if (!C_demand_2(C_SIZEOF_PAIR))
	C_rereclaim2(C_SIZEOF_PAIR * sizeof(C_word), 1);
    C_word pair = C_pair(C_heaptop, c_key, c_value);

    if (!C_demand_2(C_SIZEOF_PAIR))
	C_rereclaim2(C_SIZEOF_PAIR * sizeof(C_word), 1);
    C_word node = C_pair(C_heaptop, pair, *head);

    *head = node;
}

C_word CWordFromCFPropertyList(CFPropertyListRef plist)
{
    CFTypeID type = CFGetTypeID(plist);
    if (type  == CFStringGetTypeID())
    {
	CFDataRef utf8str = CFStringCreateExternalRepresentation(NULL, plist, kCFStringEncodingUTF8, '?');

	CFIndex len = CFDataGetLength(utf8str);

	if (!C_demand_2(C_SIZEOF_STRING(len)))
	    C_rereclaim2(C_SIZEOF_STRING(len) * sizeof(C_word), 1);
	C_word result = C_string(C_heaptop, len, (char *)CFDataGetBytePtr(utf8str));

	CFRelease(utf8str);
	return result;
    }
    if (type  == CFDataGetTypeID())
    {
	CFIndex len = CFDataGetLength(plist);

	if (!C_demand_2(C_SIZEOF_STRING(len)))
	    C_rereclaim2(C_SIZEOF_STRING(len) * sizeof(C_word), 1);
	return C_bytevector(C_heaptop, len, (char *)CFDataGetBytePtr(plist));
    }
    else if (type == CFNumberGetTypeID())
    {
	if (CFNumberIsFloatType(plist))
	{
	    double d;
	    if (CFNumberGetValue(plist, kCFNumberDoubleType, &d))
	    {
		if (!C_demand_2(C_SIZEOF_FLONUM))
		    C_rereclaim2(C_SIZEOF_FLONUM * sizeof(C_word), 1);
		return C_flonum(C_heaptop, d);
	    }
	    else
		return C_SCHEME_FALSE;
	}
	else
	{
	    int i;
	    if (CFNumberGetValue(plist, kCFNumberIntType, &i))
		return C_fix(i);
	    else
		return C_SCHEME_FALSE;
	}
    }
    else if (type == CFArrayGetTypeID())
    {
	CFIndex count = CFArrayGetCount(plist);

	C_word result = C_SCHEME_FALSE;

	if (!C_demand_2(C_SIZEOF_VECTOR(count)))
	    C_rereclaim2(C_SIZEOF_VECTOR(count) * sizeof(C_word), 1);
	result = C_vector(C_heaptop, count);

	for (int i = 0; i < count; i++)
	{
	    CFPropertyListRef pl_item = CFArrayGetValueAtIndex(plist, i);

	    C_word c_item = CWordFromCFPropertyList(pl_item);

	    if (c_item == C_SCHEME_FALSE)
		return C_SCHEME_FALSE;

	    C_set_block_item(result, i, c_item);
	}

	return result;
    }
    else if (type == CFDictionaryGetTypeID())
    {
	C_word result = C_SCHEME_END_OF_LIST;

	CFDictionaryApplyFunction(plist, add_value_key_pair, &result);

	return result;
    }
    else
	return C_SCHEME_FALSE;
}

%}

%typemap(in) CFPropertyListRef
{
    $1 = CFPropertyListFromCWord($input);

    if (!$1)
	SWIG_exception (SWIG_ValueError, "Cannot convert parameter to plist");
}

%typemap(freearg) CFPropertyListRef
{
  CFRelease($1);
}

%typemap(arginit) CFPropertyListRef
{
  $1 = NULL;
}

%typemap(out) CFPropertyListRef
{
    $result = CWordFromCFPropertyList($1);

    if ($result == C_SCHEME_FALSE)
	SWIG_exception (SWIG_ValueError, "Can't convert plist to result");
}

%exception {
    char *e;

    Try {
        $action
    }
    Catch(e) {
	SWIG_exception (SWIG_RuntimeError, e);
    }
}
