%{
#include <CoreFoundation/CoreFoundation.h>

#include "exception.h"

CFPropertyListRef CFPropertyListFromPyObject(PyObject *py_obj)
{
    if (PyUnicode_Check(py_obj))
	return CFStringCreateWithCharacters(NULL, PyUnicode_AsUnicode(py_obj), PyUnicode_GetSize(py_obj));
    else if (PyString_Check(py_obj))
	return CFDataCreate(NULL, (UInt8 *)PyString_AsString(py_obj), PyString_Size(py_obj));
    else if (PyInt_Check(py_obj))
    {
	long l = PyInt_AsLong(py_obj);
	return CFNumberCreate(kCFAllocatorDefault, kCFNumberLongType, &l);
    }
    else if (PyFloat_Check(py_obj))
    {
	double d = PyFloat_AsDouble(py_obj);
	return CFNumberCreate(kCFAllocatorDefault, kCFNumberDoubleType, &d);
    }
    else if (PySequence_Check(py_obj))
    {
	int len = PySequence_Length(py_obj);

	CFPropertyListRef *values = CFAllocatorAllocate(kCFAllocatorDefault, sizeof(CFPropertyListRef) * len, 0);
	if (!values)
	    Throw "Cannot allocate CF array for array elements";

	for (int i = 0; i < len; i++)
	{
	    PyObject *o = PySequence_GetItem(py_obj, i);
	    
	    char *e;
	    Try {
		values[i] = CFPropertyListFromPyObject(o);
	    }
	    Catch (e) {
		for (int j = 0; j < i; j++)
		    CFRelease(values[j]);
		CFAllocatorDeallocate(kCFAllocatorDefault, values);
		Throw "Cannot convert array element";
	    }
	}

	CFArrayRef a = CFArrayCreate(kCFAllocatorDefault, values, len, &kCFTypeArrayCallBacks);

	for (int i = 0; i < len; i++)
	    CFRelease(values[i]);
	CFAllocatorDeallocate(kCFAllocatorDefault, values);

	return a;
    }
    else if (PyDict_Check(py_obj))
    {
	int size = PyDict_Size(py_obj);

	CFPropertyListRef *keys = CFAllocatorAllocate(kCFAllocatorDefault, sizeof(CFPropertyListRef) * size, 0);
	CFPropertyListRef *values = CFAllocatorAllocate(kCFAllocatorDefault, sizeof(CFPropertyListRef) * size, 0);
	if (!keys || !values)
	    Throw "Cannot allocate CF array for dictionary keys or values";

	int i = 0;  /* Not clear if 'pos' is always incremented by one.  */
	PyObject *key, *value;
	int pos = 0;
	while (PyDict_Next(py_obj, &pos, &key, &value))
	{
	    if (!PyUnicode_Check(key))
	    {
		for (int j = 0; j < i; j++)
		{
		    CFRelease(keys[j]);
		    CFRelease(values[j]);
		}

		CFAllocatorDeallocate(kCFAllocatorDefault, keys);
		CFAllocatorDeallocate(kCFAllocatorDefault, values);

		Throw "Key for plist dictionary is not Unicode string";
	    }

	    char *e;
	    Try {
		keys[i] = CFPropertyListFromPyObject(key);
		
		char *e2;
		Try {
		    values[i] = CFPropertyListFromPyObject(value);
		}
		Catch (e2) {
		    CFRelease(keys[i]);
		    Throw "Cannot convert value";
		}
	    }
	    Catch (e) {
		for (int j = 0; j < i; j++)
		{
		    CFRelease(keys[j]);
		    CFRelease(values[j]);
		}

		CFAllocatorDeallocate(kCFAllocatorDefault, keys);
		CFAllocatorDeallocate(kCFAllocatorDefault, values);
		
		Throw "Cannot convert key or value in dictionary";
	    }

	    i++;
	}

	CFDictionaryRef d = CFDictionaryCreate(kCFAllocatorDefault, keys, values, size, &kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks);

	for (int i = 0; i < size; i++)
	{
	    CFRelease(keys[i]);
	    CFRelease(values[i]);
	}
	CFAllocatorDeallocate(kCFAllocatorDefault, keys);
	CFAllocatorDeallocate(kCFAllocatorDefault, values);

	return d;
    }
    else
	Throw "Unrecognized Python type for conversion into CF type";
}

PyObject *PyObjectFromCFPropertyList(CFPropertyListRef plist);

static void add_value_key_pair(const void *key, const void *value, void *d)
{
    char *e;

    PyObject *py_key = PyObjectFromCFPropertyList((CFPropertyListRef) key);
    
    Try {
	PyObject *py_value = PyObjectFromCFPropertyList((CFPropertyListRef) value);
	
	PyDict_SetItem((PyObject *)d, py_key, py_value);
	    
	Py_DECREF(py_key);
	Py_DECREF(py_value);
    }
    Catch (e) {
	Py_DECREF(py_key);
	Throw "Cannot convert value";
    }
}

PyObject *PyObjectFromCFPropertyList(CFPropertyListRef plist)
{
    CFTypeID type = CFGetTypeID(plist);
    if (type  == CFStringGetTypeID())
    {
	CFIndex len = CFStringGetLength(plist);
	const UniChar *buf = CFStringGetCharactersPtr(plist);
	if (buf)
	    return PyUnicode_FromUnicode(buf, len);
	else
	{
	    UniChar *buf2 = CFAllocatorAllocate(kCFAllocatorDefault, len * sizeof(UniChar), 0);
	    CFStringGetCharacters(plist, CFRangeMake(0, len), buf2);
	    PyObject *result = PyUnicode_FromUnicode(buf2, len);

	    CFAllocatorDeallocate(kCFAllocatorDefault, buf2);

	    return result;
	}
    }
    else if (type == CFDataGetTypeID())
	return PyString_FromStringAndSize((const char *)CFDataGetBytePtr(plist), CFDataGetLength(plist));
    else if (type == CFNumberGetTypeID())
    {
	if (CFNumberIsFloatType(plist))
	{
	    double d;
	    if (CFNumberGetValue(plist, kCFNumberDoubleType, &d))
		return PyFloat_FromDouble(d);
	    else
		Throw "Cannot get CF double value";
	}
	else
	{
	    long l;
	    if (CFNumberGetValue(plist, kCFNumberLongType, &l))
		return PyInt_FromLong(l);
	    else
		Throw "Cannot get CF long value";
	}
    }
    else if (type == CFArrayGetTypeID())
    {
	CFIndex count = CFArrayGetCount(plist);
	PyObject *l = PyList_New(count);

	for (int i = 0; i < count; i++)
	{
	    CFPropertyListRef pl_item = CFArrayGetValueAtIndex(plist, i);

	    char *e;
	    Try {
		PyObject *py_item = PyObjectFromCFPropertyList(pl_item);
		PyList_SetItem(l, i, py_item);
	    }
	    Catch(e) {
		Py_DECREF(l);
		Throw "Cannot convert array element";
	    }
	}

	return l;
    }
    else if (type == CFDictionaryGetTypeID())
    {
	PyObject *d = PyDict_New();

	char *e;
	Try {
	    CFDictionaryApplyFunction(plist, add_value_key_pair, d);
	}
	Catch (e) {
	    Py_DECREF(d);
	    Throw "Cannot convert dictionary value or key";
	}

	return d;
    }
    else
	return NULL;
}

%}

%typemap(in) CFPropertyListRef
{
    char *e;

    Try {
	$1 = CFPropertyListFromPyObject($input);
    }
    Catch(e) {
	PyErr_SetString(PyExc_ValueError, e);
	return NULL;
    }
}

%typemap(freearg) CFPropertyListRef
{
  CFRelease($1);
}

%typemap(arginit) CFPropertyListRef
{
  $1 = NULL;
}

%typemap(out) CFPropertyListRef
{
    char *e;

    Try {
	$result = PyObjectFromCFPropertyList($1);
    }
    Catch(e) {
	PyErr_SetString(PyExc_ValueError, e);
	return NULL;
    }
}

%exception {
    char *e;

    Try {
	$action
    }
    Catch(e) {
	PyErr_SetString(PyExc_ValueError, e);
	return NULL;
    }
}
