/* Copyright (c) 2005 Andrew Choi.  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.

3. All advertising materials mentioning features or use of this
   software must display the following acknowledgement:

     This product includes software developed by Andrew Choi.

4. The name "Andrew Choi" may not be used to endorse or promote
   products derived from this software without specific prior written
   permission.

THIS SOFTWARE IS PROVIDED BY ANDREW CHOI "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.  IN NO EVENT SHALL ANDREW CHOI BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.  */

#include "exception.h"
#include "MIDI.h"
#include "Device.h"

#include "Sysex.h"

Sysex *new_Sysex(CFPropertyListRef deviceDescription)
{
  /* Protect against use when client is uninitialized.  */
  if (!myMIDIClient)
    Throw "TOE must be initialized before sysex device can be created";
  
  Sysex *sysex = CFAllocatorAllocate(NULL, sizeof(Sysex), 0);
  
  CFPropertyListRef checkedDeviceDescription = CheckDeviceDescription(deviceDescription);
  
  sysex->deviceDescription = checkedDeviceDescription;
  
  EndpointPairFromDeviceDescription(checkedDeviceDescription, &sysex->receiveEndpoint, &sysex->sendEndpoint);

  if (!InitializeSysexSendRequest(&sysex->sendReq, sysex->sendEndpoint))
  {
    CFRelease(sysex->deviceDescription);
    CFAllocatorDeallocate(NULL, sysex);
    Throw "Cannot initialize send for sysex device";
  }
  
  if (!InitializeSysexReceiveRequest(&sysex->receiveReq, sysex->receiveEndpoint))
  {
    CFRelease(sysex->deviceDescription);
    CFAllocatorDeallocate(NULL, sysex);
    Throw "Cannot initialize receive for sysex device";
  }
  
  return sysex;
}

void delete_Sysex(Sysex *sysex)
{
  CFRelease(sysex->deviceDescription);
  
  TerminateSysexSendRequest(&sysex->sendReq);
  TerminateSysexReceiveRequest(&sysex->receiveReq);
  
  CFAllocatorDeallocate(NULL, sysex);
}

CFStringRef Sysex_Description(Sysex *sysex)
{
  SInt32 sendUID, receiveUID;
  MIDIObjectGetIntegerProperty(sysex->sendEndpoint, kMIDIPropertyUniqueID, &sendUID);
  MIDIObjectGetIntegerProperty(sysex->receiveEndpoint, kMIDIPropertyUniqueID, &receiveUID);
  
  return CFStringCreateWithFormat(NULL, NULL, CFSTR("<Sysex device '%@', sendEndpoint'%d', receiveEndpoint '%d'>"), sysex->deviceDescription, sendUID, receiveUID);
}

void Sysex_Send(Sysex *sysex, CFPropertyListRef sysexData)
{
  if (CFGetTypeID(sysexData) != CFDataGetTypeID())
    Throw "Sysex data to send is not of CFDataRef type";
  
  SendSysexData(&sysex->sendReq, (CFDataRef) sysexData);
}

int Sysex_NumOfBytesSent(Sysex *sysex)
{
  return NumOfBytesSent(&sysex->sendReq);
}

void Sysex_CancelSend(Sysex *sysex)
{
  CancelSend(&sysex->sendReq);
}

CFPropertyListRef Sysex_Receive(Sysex *sysex)
{
  return (CFPropertyListRef) ReceiveSysexData(&sysex->receiveReq);
}

int Sysex_NumOfBytesReceived(Sysex *sysex)
{
  return NumOfBytesReceived(&sysex->receiveReq);
}

