/* Copyright (c) 2005 Andrew Choi.  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.

3. All advertising materials mentioning features or use of this
   software must display the following acknowledgement:

     This product includes software developed by Andrew Choi.

4. The name "Andrew Choi" may not be used to endorse or promote
   products derived from this software without specific prior written
   permission.

THIS SOFTWARE IS PROVIDED BY ANDREW CHOI "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.  IN NO EVENT SHALL ANDREW CHOI BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.  */

#include <CoreMIDI/MIDIServices.h>
#include <Foundation/Foundation.h>

#include "MIDI.h"

#include "Device.h"

CFDataRef ConnectionsFromEndpoint(MIDIEndpointRef endpoint)
{
  CFDataRef connections;
  
  if (MIDIObjectGetDataProperty(endpoint, kMIDIPropertyConnectionUniqueID, &connections) != noErr)
    return NULL;
  
  /* CoreMIDI may return a connections array with zero length! */
  if (CFDataGetLength(connections) == 0)
  {
    CFRelease(connections);
    return NULL;
  }
  
  return connections;
}

MIDIDeviceRef DeviceFromEndpoint(MIDIEndpointRef endpoint)
{
  MIDIEntityRef entity;
  if (MIDIEndpointGetEntity(endpoint, &entity) != noErr)
    return NULL;
  
  MIDIDeviceRef device;
  if (MIDIEntityGetDevice(entity, &device) != noErr)
    return NULL;

  return device;
}

static CFStringRef NameFromObject(MIDIObjectRef objectRef)
{
  CFStringRef name;
  if (MIDIObjectGetStringProperty(objectRef, kMIDIPropertyName, &name) != noErr)
    return NULL;
  
  return name;
}

static CFStringRef NameFromDeviceAndEndpoint(MIDIDeviceRef device, MIDIEndpointRef endpoint)
{
  CFStringRef deviceName = NameFromObject(device);
  if (!deviceName)
    return NULL;
  
  CFStringRef endpointName = NameFromObject(endpoint);
  if (!endpointName)
  {
    CFRelease(deviceName);
    return NULL;
  }
  
  /* If device name is a prefix of endpoint name, simply use endpoint name.  */
  if (CFStringCompareWithOptions(endpointName, deviceName, CFRangeMake(0, CFStringGetLength(deviceName)), 0) == kCFCompareEqualTo)
  {
    CFRelease(deviceName);  
    return endpointName;
  }
  else
  {
    CFStringRef name = CFStringCreateWithFormat(NULL, NULL, CFSTR("%@ %@"), deviceName, endpointName);
    CFRelease(deviceName);
    CFRelease(endpointName);
    return name;
  }
}

static CFPropertyListRef NameFromUnconnectedEndpoint(MIDIEndpointRef endpoint)
{
  MIDIDeviceRef device = DeviceFromEndpoint(endpoint);

  if (!device)
    return NameFromObject(endpoint);
  else
    return NameFromDeviceAndEndpoint(device, endpoint);
}

static CFStringRef NameFromExternalDeviceAndEndpoint(MIDIDeviceRef device, MIDIEndpointRef endpoint)
{
  CFStringRef deviceName = NameFromObject(device);
  if (!deviceName)
    return NULL;
  
  /* Use just the device name if it has only one entity.  */
  if (MIDIDeviceGetNumberOfEntities(device) <= 1)
    return deviceName;
  
  CFStringRef endpointName = NameFromObject(endpoint);
  if (!endpointName)
  {
    CFRelease(deviceName);
    return NULL;
  }
  
  CFStringRef name = CFStringCreateWithFormat(NULL, NULL, CFSTR("%@ %@"), deviceName, endpointName);
  CFRelease(deviceName);
  CFRelease(endpointName);
  return name;
}

static CFDictionaryRef ManufacturerModelDeviceIDFromDevice(MIDIDeviceRef device)
{
  CFStringRef keys[3] = { CFSTR("Manufacturer"), CFSTR("Model"), CFSTR("DeviceID") };
  CFPropertyListRef values[3];

  CFDictionaryRef dict = NULL;
  
  if (MIDIObjectGetStringProperty(device, kMIDIPropertyManufacturer, (CFStringRef *) &values[0]) != noErr)
    goto fail1;
  
  if (MIDIObjectGetStringProperty(device, kMIDIPropertyModel, (CFStringRef *) &values[1]) != noErr)
    goto fail2;

  SInt32 deviceID;
  OSStatus s = MIDIObjectGetIntegerProperty(device, kMIDIPropertyDeviceID, &deviceID);

  if (s != noErr)
  {
    dict = CFDictionaryCreate(NULL, (const void **) keys, (const void **) values, 2, &kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks);
  }
  else
  {
    values[2] = CFNumberCreate(NULL, kCFNumberSInt32Type, &deviceID);

    dict = CFDictionaryCreate(NULL, (const void **) keys, (const void **) values, 3, &kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks);

    CFRelease(values[2]);
  }
    
  CFRelease(values[1]);
fail2:
  CFRelease(values[0]);
fail1:
  return dict;
}
							
static CFPropertyListRef DeviceDescriptionFromEndpointOrDevice(MIDIObjectRef objectRef, MIDIObjectType objectType)
{
  if (objectType == kMIDIObjectType_ExternalSource || objectType == kMIDIObjectType_ExternalDestination)
  {
    /* Connected to an external device's endpoint (10.3 and later). */
    MIDIEndpointRef endpoint = (MIDIEndpointRef) objectRef;
    MIDIDeviceRef device = DeviceFromEndpoint(endpoint);
    
    if (!device)
      @throw EXCEPTION(@"External source or destination must be contained in a device");

    CFStringRef displayName = NameFromExternalDeviceAndEndpoint(device, endpoint);
    
    CFDictionaryRef dict = ManufacturerModelDeviceIDFromDevice(device);
    CFMutableDictionaryRef dict2 = CFDictionaryCreateMutableCopy(NULL, 0, dict);
    CFRelease(dict);
    
    CFDictionaryAddValue(dict2, CFSTR("DisplayName"), displayName);
    
    CFRelease(displayName);
    
    return dict2;
  }
  else if (objectType == kMIDIObjectType_ExternalDevice)
  {
    // Connected to an external device (10.2)
    MIDIDeviceRef device = (MIDIDeviceRef) objectRef;
    
    CFStringRef displayName = NameFromObject(device);

    CFDictionaryRef dict = ManufacturerModelDeviceIDFromDevice(device);
    CFMutableDictionaryRef dict2 = CFDictionaryCreateMutableCopy(NULL, 0, dict);
    CFRelease(dict);
    
    CFDictionaryAddValue(dict2, CFSTR("DisplayName"), displayName);
    
    CFRelease(displayName);
    
    return dict2;
  }
  else
    @throw EXCEPTION(@"Endpoint connected to neither endpoint in external device nor external device");
}

static CFMutableArrayRef DeviceDescriptionFromConnections(CFDataRef connections)
{
  CFMutableArrayRef deviceDescriptionArray = CFArrayCreateMutable(NULL, 0, &kCFTypeArrayCallBacks);
  
  CFIndex n = CFDataGetLength(connections) / sizeof(MIDIUniqueID);
  for (int i = 0; i < n; i++) {
    MIDIUniqueID uid;
    CFDataGetBytes(connections, CFRangeMake(i * sizeof(MIDIUniqueID), sizeof(MIDIUniqueID)), (UInt8 *) &uid);

    MIDIObjectRef objectRef;
    MIDIObjectType objectType;
    OSStatus s = MIDIObjectFindByUniqueID(uid, &objectRef, &objectType);
    if (s == noErr)
    {
      CFDictionaryRef deviceDescription = DeviceDescriptionFromEndpointOrDevice(objectRef, objectType);
      CFArrayAppendValue(deviceDescriptionArray, deviceDescription);
      CFRelease(deviceDescription);
    }
  }
  
  return deviceDescriptionArray;
}

static CFDictionaryRef CreateDictionaryWithKeyAndValue(CFStringRef key, CFStringRef value)
{
  return CFDictionaryCreate(NULL, (const void **) &key, (const void **) &value, 1, &kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks);
}

static CFComparisonResult CompareDisplayNames(const void *val1, const void *val2, void *context)
{
  const CFDictionaryRef v1 = (CFDictionaryRef) val1;
  const CFDictionaryRef v2 = (CFDictionaryRef) val2;
  
  return CFStringCompare(CFDictionaryGetValue(v1, CFSTR("DisplayName")), CFDictionaryGetValue(v2, CFSTR("DisplayName")), 0);
}

static CFStringRef DisplayNameFromDeviceDescriptionArray(CFArrayRef deviceDescriptionArray)
{
  CFMutableArrayRef displayNameArray = CFArrayCreateMutable(NULL, CFArrayGetCount(deviceDescriptionArray), &kCFTypeArrayCallBacks);
  
  for (int i = 0; i < CFArrayGetCount(deviceDescriptionArray); i++)
    CFArrayAppendValue(displayNameArray, CFDictionaryGetValue(CFArrayGetValueAtIndex(deviceDescriptionArray, i), CFSTR("DisplayName")));

  CFStringRef displayName = CFStringCreateByCombiningStrings(NULL, displayNameArray, CFSTR(", "));
  
  CFRelease(displayNameArray);
  
  return displayName;
}

static CFPropertyListRef DeviceDescriptionFromEndpoint(MIDIEndpointRef endpoint)
{
  CFDataRef connections = ConnectionsFromEndpoint(endpoint);
  if (!connections)
  {
    CFStringRef name = NameFromUnconnectedEndpoint(endpoint);
    CFDictionaryRef dict = CreateDictionaryWithKeyAndValue(CFSTR("DisplayName"), name);
    CFRelease(name);
    return dict;
  }
  else
  {
    CFMutableArrayRef deviceDescription = DeviceDescriptionFromConnections(connections);
    CFRelease(connections);
    
    CFArraySortValues(deviceDescription, CFRangeMake(0, CFArrayGetCount(deviceDescription)), CompareDisplayNames, 0);
    
    CFStringRef displayName = DisplayNameFromDeviceDescriptionArray(deviceDescription);
    
    CFStringRef keys[2] = { CFSTR("DisplayName"), CFSTR("Devices") };
    CFPropertyListRef values[2] = { displayName, deviceDescription };
    
    CFDictionaryRef dict = CFDictionaryCreate(NULL, (const void **) keys, (const void **) values, 2, &kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks);

    CFRelease(displayName);
    CFRelease(deviceDescription);
    
    return dict;
  }
}

static CFMutableArrayRef sources = NULL;
static CFMutableArrayRef destinations = NULL;
static CFMutableArrayRef pairs = NULL;

/* Whether these arrays are allocated depends on whether the corresponding arrays above are allocated, respectively.  */
static MIDIEndpointRef *sourceEndpoints;
static MIDIEndpointRef *destinationEndpoints;
static MIDIEndpointRef *pairSourceEndpoints;
static MIDIEndpointRef *pairDestinationEndpoints;

/* Called from MIDI.m.  */
void UpdateDeviceTables()
{
  if (sources)
  {
    CFRelease(sources);
    sources = NULL;

    CFAllocatorDeallocate(NULL, sourceEndpoints);
  }

  if (destinations)
  {
    CFRelease(destinations);
    destinations = NULL;

    CFAllocatorDeallocate(NULL, destinationEndpoints);
  }

  if (pairs)
  {
    CFRelease(pairs);
    pairs = NULL;
    
    CFAllocatorDeallocate(NULL, pairSourceEndpoints);
    CFAllocatorDeallocate(NULL, pairDestinationEndpoints);
  }
}

static CFMutableArrayRef GetSources()
{
  CheckMIDISetupChange();
  
  if (!sources)
  {
    int n = MIDIGetNumberOfSources();

    sources = CFArrayCreateMutable(NULL, n, &kCFTypeArrayCallBacks);
    sourceEndpoints = CFAllocatorAllocate(NULL, sizeof(MIDIEndpointRef) * n, 0);
    
    for (int i = 0; i < n; i++)
    {
      MIDIEndpointRef endpoint = MIDIGetSource(i);
      if (endpoint)
      {
	CFPropertyListRef deviceDescription = DeviceDescriptionFromEndpoint(endpoint);
	CFArrayAppendValue(sources, deviceDescription);
	CFRelease(deviceDescription);
	
	sourceEndpoints[i] = endpoint;
      }
    }
  }
  
  return sources;
}

static CFMutableArrayRef GetDestinations()
{
  CheckMIDISetupChange();
  
  if (!destinations)
  {
    int n = MIDIGetNumberOfDestinations();

    destinations = CFArrayCreateMutable(NULL, n, &kCFTypeArrayCallBacks);
    destinationEndpoints = CFAllocatorAllocate(NULL, sizeof(MIDIEndpointRef) * n, 0);
    
    for (int i = 0; i < n; i++)
    {
      MIDIEndpointRef endpoint = MIDIGetDestination(i);
      if (endpoint)
      {
	CFPropertyListRef deviceDescription = DeviceDescriptionFromEndpoint(endpoint);
	CFArrayAppendValue(destinations, deviceDescription);
	CFRelease(deviceDescription);
	
	destinationEndpoints[i] = endpoint;
      }      
    }
  }
  
  return destinations;
}

static CFMutableArrayRef GetPairs()
{
  CFArrayRef mySources = GetSources();
  CFArrayRef myDestinations = GetDestinations();
  
  if (!pairs)
  {
    pairs = CFArrayCreateMutable(NULL, 0, &kCFTypeArrayCallBacks);
    
    int n = CFArrayGetCount(myDestinations);

    for (int i = 0; i < n; i++)
    {
      CFPropertyListRef dest = CFArrayGetValueAtIndex(myDestinations, i);
      
      if (CFArrayContainsValue(mySources, CFRangeMake(0, CFArrayGetCount(mySources)), dest))
	CFArrayAppendValue(pairs, dest);
    }
    
    int n2 = CFArrayGetCount(pairs);
    pairSourceEndpoints = CFAllocatorAllocate(NULL, sizeof(MIDIEndpointRef) * n2, 0);
    pairDestinationEndpoints = CFAllocatorAllocate(NULL, sizeof(MIDIEndpointRef) * n2, 0);
    for (int i = 0; i < n2; i++)
    {
      CFPropertyListRef deviceDescription = CFArrayGetValueAtIndex(pairs, i);
      
      CFIndex j = CFArrayGetFirstIndexOfValue(mySources, CFRangeMake(0, CFArrayGetCount(mySources)), deviceDescription);
      CFIndex k = CFArrayGetFirstIndexOfValue(myDestinations, CFRangeMake(0, CFArrayGetCount(myDestinations)), deviceDescription);
      
      if (j == -1 || k == -1)
	@throw EXCEPTION(@"Can't find device description when constructing pair endpoint table");
      
      pairSourceEndpoints[i] = sourceEndpoints[j];
      pairDestinationEndpoints[i] = destinationEndpoints[k];
    }
  }
  
  return pairs;
}

static Boolean MatchManufacturerModelDeviceID(CFPropertyListRef p1, CFPropertyListRef p2)
{
  if (CFGetTypeID(p1) != CFDictionaryGetTypeID() || CFGetTypeID(p2) != CFDictionaryGetTypeID())
    return false;
    
  CFArrayRef dda1;
  CFArrayRef dda2;
  if (!CFDictionaryGetValueIfPresent(p1, CFSTR("Devices"), (const void **) &dda1))
    return false;
  if (!CFDictionaryGetValueIfPresent(p2, CFSTR("Devices"), (const void **) &dda2))
    return false;
  if (CFGetTypeID(dda1) != CFArrayGetTypeID() || CFGetTypeID(dda2) != CFArrayGetTypeID())
    return false;

  if (CFArrayGetCount(dda1) != 1 || CFArrayGetCount(dda2) != 1)
    return false;
  
  CFDictionaryRef dd1 = CFArrayGetValueAtIndex(dda1, 0);
  CFDictionaryRef dd2 = CFArrayGetValueAtIndex(dda2, 0);
  if (CFGetTypeID(dd1) != CFDictionaryGetTypeID() || CFGetTypeID(dd1) != CFDictionaryGetTypeID())
    return false;
  
  CFStringRef manufacturer1;
  CFStringRef manufacturer2;
  if (!CFDictionaryGetValueIfPresent(dd1, CFSTR("Manufacturer"), (const void **) &manufacturer1))
    return false;
  if (!CFDictionaryGetValueIfPresent(dd2, CFSTR("Manufacturer"), (const void **) &manufacturer2))
    return false;
  if (CFGetTypeID(manufacturer1) != CFStringGetTypeID() || CFGetTypeID(manufacturer2) != CFStringGetTypeID())
    return false;
  
  CFStringRef model1;
  CFStringRef model2;
  if (!CFDictionaryGetValueIfPresent(dd1, CFSTR("Model"), (const void **) &model1))
    return false;
  if (!CFDictionaryGetValueIfPresent(dd2, CFSTR("Model"), (const void **) &model2))
    return false;
  if (CFGetTypeID(model1) != CFStringGetTypeID() || CFGetTypeID(model2) != CFStringGetTypeID())
    return false;
  
  CFNumberRef deviceID1;
  CFNumberRef deviceID2;
  if (!CFDictionaryGetValueIfPresent(dd1, CFSTR("DeviceID"), (const void **) &deviceID1))
    return false;
  if (!CFDictionaryGetValueIfPresent(dd2, CFSTR("DeviceID"), (const void **) &deviceID2))
    return false;
  if (CFGetTypeID(deviceID1) != CFNumberGetTypeID() || CFGetTypeID(deviceID2) != CFNumberGetTypeID())
    return false;
  
  return CFEqual(manufacturer1, manufacturer2) && CFEqual(model1, model2) && CFEqual(deviceID1, deviceID2);
}

static Boolean MatchManufacturerModel(CFPropertyListRef p1, CFPropertyListRef p2)
{
  if (CFGetTypeID(p1) != CFDictionaryGetTypeID() || CFGetTypeID(p2) != CFDictionaryGetTypeID())
    return false;
  
  CFArrayRef dda1;
  CFArrayRef dda2;
  if (!CFDictionaryGetValueIfPresent(p1, CFSTR("Devices"), (const void **) &dda1))
    return false;
  if (!CFDictionaryGetValueIfPresent(p2, CFSTR("Devices"), (const void **) &dda2))
    return false;
  if (CFGetTypeID(dda1) != CFArrayGetTypeID() || CFGetTypeID(dda2) != CFArrayGetTypeID())
    return false;
  
  if (CFArrayGetCount(dda1) != 1 || CFArrayGetCount(dda2) != 1)
    return false;
  
  CFDictionaryRef dd1 = CFArrayGetValueAtIndex(dda1, 0);
  CFDictionaryRef dd2 = CFArrayGetValueAtIndex(dda2, 0);
  if (CFGetTypeID(dd1) != CFDictionaryGetTypeID() || CFGetTypeID(dd1) != CFDictionaryGetTypeID())
    return false;
  
  CFStringRef manufacturer1;
  CFStringRef manufacturer2;
  if (!CFDictionaryGetValueIfPresent(dd1, CFSTR("Manufacturer"), (const void **) &manufacturer1))
    return false;
  if (!CFDictionaryGetValueIfPresent(dd2, CFSTR("Manufacturer"), (const void **) &manufacturer2))
    return false;
  if (CFGetTypeID(manufacturer1) != CFStringGetTypeID() || CFGetTypeID(manufacturer2) != CFStringGetTypeID())
    return false;
  
  CFStringRef model1;
  CFStringRef model2;
  if (!CFDictionaryGetValueIfPresent(dd1, CFSTR("Model"), (const void **) &model1))
    return false;
  if (!CFDictionaryGetValueIfPresent(dd2, CFSTR("Model"), (const void **) &model2))
    return false;
  if (CFGetTypeID(model1) != CFStringGetTypeID() || CFGetTypeID(model2) != CFStringGetTypeID())
    return false;
  
  return CFEqual(manufacturer1, manufacturer2) && CFEqual(model1, model2);
}

static Boolean MatchDisplayName(CFPropertyListRef p1, CFPropertyListRef p2)
{
  if (CFGetTypeID(p1) != CFDictionaryGetTypeID() || CFGetTypeID(p2) != CFDictionaryGetTypeID())
    return false;
  
  CFStringRef displayName1;
  CFStringRef displayName2;
  if (!CFDictionaryGetValueIfPresent(p1, CFSTR("DisplayName"), (const void **) &displayName1))
    return false;
  if (!CFDictionaryGetValueIfPresent(p2, CFSTR("DisplayName"), (const void **) &displayName2))
    return false;
  if (CFGetTypeID(displayName1) != CFStringGetTypeID() || CFGetTypeID(displayName2) != CFStringGetTypeID())
    return false;
    
  return CFEqual(displayName1, displayName2);
}

typedef Boolean (*MatchFunction) (CFPropertyListRef p1, CFPropertyListRef p2);

static CFIndex ResolveName(MatchFunction f, CFPropertyListRef name, CFArrayRef deviceDescriptionArray)
{
  CFIndex n = CFArrayGetCount(deviceDescriptionArray);
  
  CFIndex foundAt = -1;
  
  for (CFIndex i = 0; i < n; i++)
  {
    if ((*f)(name, CFArrayGetValueAtIndex(deviceDescriptionArray, i)))
    {
      if (foundAt != -1)
      {
	foundAt = -2;  /* -1 = no match, -2 = multiple matches.  Not distinguished elsewhere yet.  */
	break;
      }
      foundAt = i;
    }
  }
  
  return foundAt;
}

static CFIndex ResolveNameInDeviceDescriptionArray(CFPropertyListRef name, CFArrayRef deviceDescriptionArray)
{
  CFIndex foundAt = ResolveName(CFEqual, name, deviceDescriptionArray);
  if (foundAt >= 0)
    return foundAt;

  foundAt = ResolveName(MatchManufacturerModelDeviceID, name, deviceDescriptionArray);
  if (foundAt >= 0)
    return foundAt;
  
  foundAt = ResolveName(MatchManufacturerModel, name, deviceDescriptionArray);
  if (foundAt >= 0)
    return foundAt;
  
  foundAt = ResolveName(MatchDisplayName, name, deviceDescriptionArray);
  
  return foundAt;
}

void EndpointPairFromDeviceDescription(CFPropertyListRef deviceDescription, MIDIEndpointRef *source, MIDIEndpointRef *destination)
{
  CFArrayRef myPairs = GetPairs();
  
  CFIndex i = ResolveNameInDeviceDescriptionArray(deviceDescription, myPairs);
  
  if (i < 0)
    @throw EXCEPTION(@"Device description for device pair not found");
  
  *source = pairSourceEndpoints[i];
  *destination = pairDestinationEndpoints[i];
}

MIDIEndpointRef DestinationEndpointFromDeviceDescription(CFPropertyListRef deviceDescription)
{
  CFArrayRef myDestinations = GetDestinations();
  
  CFIndex i = ResolveNameInDeviceDescriptionArray(deviceDescription, myDestinations);
  
  if (i < 0)
    @throw EXCEPTION(@"Device description for destination not found");
  
  return destinationEndpoints[i];
}

CFPropertyListRef CheckDeviceDescription(CFPropertyListRef deviceDescription)
{
  CFTypeID type = CFGetTypeID(deviceDescription);
  
  if (type == CFDictionaryGetTypeID())
  {
    if (!CFDictionaryContainsKey(deviceDescription, CFSTR("DisplayName")))
      @throw EXCEPTION(@"Device description does not contain value for key DisplayName");
    
    CFStringRef displayName = CFDictionaryGetValue(deviceDescription, CFSTR("DisplayName"));
    if (CFGetTypeID(displayName) != CFStringGetTypeID())
      @throw EXCEPTION(@"DisplayName value in device description is not of type CFStringRef");
    
    return CFRetain(deviceDescription);
  }
  else if (type == CFStringGetTypeID())
  {
    CFStringRef keys[] = { CFSTR("DisplayName") };
    CFPropertyListRef values[] = { deviceDescription };
    
    return CFDictionaryCreate(NULL, (const void **) keys, (const void **) values, 1, &kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks);
  }
  else if (type == CFDataGetTypeID())
  {
    CFStringRef keys[] = { CFSTR("DisplayName") };
    CFStringRef stringDisplayName = CFStringCreateWithBytes(NULL, CFDataGetBytePtr(deviceDescription), CFDataGetLength(deviceDescription), kCFStringEncodingUTF8, false);
    CFPropertyListRef values[] = { stringDisplayName };
    
    CFDictionaryRef newDeviceDescription = CFDictionaryCreate(NULL, (const void **) keys, (const void **) values, 1, &kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks);
    
    CFRelease(stringDisplayName);
    
    return newDeviceDescription;
  }
  else
    @throw EXCEPTION(@"Device description is not of type CFDictionaryRef, CFStringRef, or CFDataRef");
}

CFPropertyListRef Devices(int types)
{
  if (types == kDeviceTypeSources)
    return CFRetain(GetSources());
  else if (types == kDeviceTypeDestinations)
    return CFRetain(GetDestinations());
  else if (types == kDeviceTypePairs)
    return CFRetain(GetPairs());
  else
    @throw EXCEPTION(@"Unknown device type");
}
