/* Copyright (c) 2005 Andrew Choi.  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.

3. All advertising materials mentioning features or use of this
   software must display the following acknowledgement:

     This product includes software developed by Andrew Choi.

4. The name "Andrew Choi" may not be used to endorse or promote
   products derived from this software without specific prior written
   permission.

THIS SOFTWARE IS PROVIDED BY ANDREW CHOI "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.  IN NO EVENT SHALL ANDREW CHOI BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.  */

#include <CoreMIDI/MIDIServices.h>
#include <Foundation/Foundation.h>

#import "AliasUtilities.h"

#include "Device.h"

#include "PatchNames.h"

static MIDIDeviceRef ExternalDeviceFromDestinationEndpoint(MIDIEndpointRef destinationEndpoint)
{
  CFDataRef connections = ConnectionsFromEndpoint(destinationEndpoint);
  
  if (!connections || CFDataGetLength(connections) != sizeof(MIDIUniqueID))
    @throw EXCEPTION(@"Destination endpoint is unconnected or connected to multiple external devices");
  
  MIDIUniqueID uid;
  CFDataGetBytes(connections, CFRangeMake(0, sizeof(MIDIUniqueID)), (UInt8 *) &uid);
  CFRelease(connections);
  
  MIDIObjectRef objectRef;
  MIDIObjectType objectType;
  if (MIDIObjectFindByUniqueID(uid, &objectRef, &objectType) != noErr)
    @throw EXCEPTION(@"External device for destination cannot be located");
  
  MIDIDeviceRef device;
  if (objectType == kMIDIObjectType_ExternalSource || objectType == kMIDIObjectType_ExternalDestination)
  {
    /* Connected to an external device's endpoint (10.3 and later). */
    MIDIEndpointRef endpoint = (MIDIEndpointRef) objectRef;
    device = DeviceFromEndpoint(endpoint);
    
    if (!device)
      @throw EXCEPTION(@"External endpoint connected to destination not contained in a device");
  }
  else if (objectType == kMIDIObjectType_ExternalDevice)
  {
    // Connected to an external device (10.2)
    device = (MIDIDeviceRef) objectRef;
  }
  else
    @throw EXCEPTION(@"Destination endpoint not connected to external endpoint or external device");
  
  return device;
}

static NSXMLDocument *XMLDocumentFromFile(NSString *filename)
{
  if ([filename length] == 0)
    @throw EXCEPTION(@"Master name document filename cannot be empty");
  
  NSError *err = nil;
  
  NSURL *furl = [NSURL fileURLWithPath:filename];
  if (!furl)
    @throw EXCEPTION(@"Can't create URL from filename");
  
  NSXMLDocument *xmlDoc = [[NSXMLDocument alloc] initWithContentsOfURL:furl options:(NSXMLNodePreserveWhitespace|NSXMLNodePreserveCDATA) error:&err];
  if (xmlDoc == nil)
    xmlDoc = [[NSXMLDocument alloc] initWithContentsOfURL:furl options:NSXMLDocumentTidyXML error:&err];
  
  if (xmlDoc == nil || err)
    @throw EXCEPTION(@"Master name document format error");
  
  return [xmlDoc autorelease];
}

static NSXMLDocument *XMLDocumentFromDevice(MIDIDeviceRef deviceRef)
{
  CFDictionaryRef cfNameConfiguration;
  if (MIDIObjectGetDictionaryProperty(deviceRef, kMIDIPropertyNameConfiguration, &cfNameConfiguration) != noErr)
    @throw EXCEPTION(@"Destination device does not have name configuration property");
  
  CFDataRef masterAliasData = (CFDataRef)[(NSDictionary *)cfNameConfiguration objectForKey:@"master"];
  
  [(NSDictionary *)cfNameConfiguration autorelease];
  
  CFStringRef pathname = POSIXPathnameFromAliasData(masterAliasData);
  
  NSXMLDocument *xmlDoc = XMLDocumentFromFile((NSString *) pathname);
  
  CFRelease(pathname);
  
  return xmlDoc;
}

static NSArray *CurrentChannelNameSetsFromDevice(MIDIDeviceRef deviceRef)
{
  CFDictionaryRef cfNameConfiguration;
  if (MIDIObjectGetDictionaryProperty(deviceRef, kMIDIPropertyNameConfiguration, &cfNameConfiguration) != noErr)
    @throw EXCEPTION(@"Destination device does not have name configuration property");
  
  CFArrayRef currentChannelNameSets = (CFArrayRef)[(NSDictionary *)cfNameConfiguration objectForKey:@"currentChannelNameSets"];
  
  NSArray *currentChannelNameSetsCopy = [NSArray arrayWithArray:(NSArray *) currentChannelNameSets];
  
  CFRelease(cfNameConfiguration);
  
  return currentChannelNameSetsCopy;
}

static NSArray *BanksForNameSetFromXMLDocument(NSXMLDocument *xmlDoc, NSString *nameSet)
{
  NSError *err = nil;
  
  NSArray *channelNameSets = [xmlDoc nodesForXPath:[NSString stringWithFormat:@"./MIDINameDocument/MasterDeviceNames/ChannelNameSet[@Name=\"%@\"]", nameSet] error:&err];
  if (channelNameSets == nil || err)
    @throw EXCEPTION(@"Cannot get channel name set from MIDI name document");
  if ([channelNameSets count] != 1)
    @throw EXCEPTION(@"More than one channel name sets with the same name in MIDI name document");
  
  NSXMLElement *channelNameSet = [channelNameSets objectAtIndex:0];

  NSArray *names = [channelNameSet nodesForXPath:@"./PatchBank/@Name" error:&err];
  if (names == nil || err)
    @throw EXCEPTION(@"No patch bank names for the given channel name set in MIDI name document");

  NSMutableArray *banks = [NSMutableArray arrayWithCapacity:0];

  for (int i = 0; i < [names count]; i++)
    [banks addObject:[[names objectAtIndex:i] objectValue]];

  return banks;
}

CFPropertyListRef Destination_Banks(Destination *destination, int channel)
{
  if (channel < 1 || channel > 16)
    @throw EXCEPTION(@"MIDI channel is not between 1 and 16");
  
  MIDIDeviceRef device = ExternalDeviceFromDestinationEndpoint(destination->endpoint);
  
  NSArray *currentChannelNameSets = CurrentChannelNameSetsFromDevice(device);
  
  NSXMLDocument *masterNameDocumentXMLDocument = XMLDocumentFromDevice(device);
  
  NSArray *banks = BanksForNameSetFromXMLDocument(masterNameDocumentXMLDocument, [currentChannelNameSets objectAtIndex:channel - 1]);
  
  /* Ownership of CFPropertyListRef return value is always passed to caller.  NSObject return values follow Objective C convention and are autoreleased.  */
  CFArrayRef banks2 = CFArrayCreateCopy(NULL, (CFArrayRef) banks);
  
  return banks2;
}

/* Determine whether "bank" for "device" has an overriding patch name list.  If so, return the overriding XML document and name of the overriding patch name list in it.  */
static Boolean OverridingFileAndPatchNameListForBankFromDevice(MIDIDeviceRef device, NSString *bank, NSXMLDocument **overridingNameDocumentXMLDocument, NSString **patchNameList)
{
  CFDictionaryRef cfNameConfiguration;
  if (MIDIObjectGetDictionaryProperty(device, kMIDIPropertyNameConfiguration, &cfNameConfiguration) != noErr)
    @throw EXCEPTION(@"Destination device does not have name configuration property");
  
  CFDictionaryRef banks = (CFDictionaryRef)[(NSDictionary *)cfNameConfiguration objectForKey:@"banks"];
  if (!banks)
  {
    CFRelease(cfNameConfiguration);
    return false;
  }
  
  CFDictionaryRef overrideInfo = (CFDictionaryRef)[(NSDictionary *)banks objectForKey:bank];
  if (!overrideInfo)
  {
    CFRelease(cfNameConfiguration);
    return false;
  }
  
  CFDataRef file = (CFDataRef)[(NSDictionary *)overrideInfo objectForKey:@"file"];
  NSString *patchNameList2 = (NSString *)[(NSDictionary *)overrideInfo objectForKey:@"patchNameList"];

  if (!file || !patchNameList2)
  {
    CFRelease(cfNameConfiguration);
    return false;
  }

  CFStringRef pathname = POSIXPathnameFromAliasData(file);
  *overridingNameDocumentXMLDocument = XMLDocumentFromFile((NSString *) pathname);
  CFRelease(pathname);

  /* patchNameList2 will not stay around when cfNameConfiguration is released.  */
  *patchNameList = [NSString stringWithString:patchNameList2];
  
  CFRelease(cfNameConfiguration);
  return true;
}

static NSXMLElement *PatchNameListFromXMLDocument(NSXMLDocument *xmlDoc, NSString *patchNameList)
{
  NSError *err = nil;
  NSArray *patchNameLists = [xmlDoc nodesForXPath:[NSString stringWithFormat:@"./MIDINameDocument/MasterDeviceNames/ChannelNameSet/PatchBank[@Name=\"%@\"]/PatchNameList", patchNameList] error:&err];
  if (!err && patchNameLists != nil && [patchNameLists count] == 1)
    return [patchNameLists objectAtIndex:0];

  /* Try alternative format.  */
  err = nil;
  patchNameLists = [xmlDoc nodesForXPath:[NSString stringWithFormat:@"./MIDINameDocument/MasterDeviceNames/PatchNameList[@Name=\"%@\"]", patchNameList] error:&err];
  if (!err && patchNameLists != nil && [patchNameLists count] == 1)
    return [patchNameLists objectAtIndex:0];
    
  @throw EXCEPTION(@"Cannot get patch name list from MIDI name document");
}

static NSArray *PatchesFromPatchNameList(NSXMLElement *patchNameListXMLElement)
{
  NSError *err = nil;
  NSArray *names = [patchNameListXMLElement nodesForXPath:@"./Patch/@Name" error:&err];
  if (names && [names count] > 0 && !err)
  {  
    NSMutableArray *patches = [NSMutableArray arrayWithCapacity:0];
  
    for (int i = 0; i < [names count]; i++)
      [patches addObject:[[names objectAtIndex:i] objectValue]];
  
    return patches;
  }
    
  @throw EXCEPTION(@"Cannot get patch names from MIDI name document");
}

static NSXMLElement *PatchNameListForBankFromDevice(MIDIDeviceRef device, NSString *bank)
{
  NSXMLDocument *overridingNameDocumentXMLDocument;
  NSString *patchNameList;

  if (OverridingFileAndPatchNameListForBankFromDevice(device, bank, &overridingNameDocumentXMLDocument, &patchNameList))
  {
    return PatchNameListFromXMLDocument(overridingNameDocumentXMLDocument, patchNameList);
  }
  else
  {
    NSXMLDocument *masterNameDocumentXMLDocument = XMLDocumentFromDevice(device);
    
    return PatchNameListFromXMLDocument(masterNameDocumentXMLDocument, bank);    
  }
}

static NSData *ProgramChangeForPatchFromPatchNameList(NSXMLElement *patchNameListXMLElement, int channel, NSString *patch)
{
  NSError *err = nil;
  NSArray *programChanges = [patchNameListXMLElement nodesForXPath:[NSString stringWithFormat:@"./Patch[@Name=\"%@\"]/@ProgramChange", patch] error:&err];
  if (!err && programChanges != nil && [programChanges count] == 1)
  {    
    NSXMLNode *programChange = [programChanges objectAtIndex:0];
    
    char MIDIProgramChange[2];
    MIDIProgramChange[0] = 0xc0 | ((channel - 1) & 0xf);
    MIDIProgramChange[1] = [[programChange objectValue] intValue];
    
    return [NSData dataWithBytes:MIDIProgramChange length:2];
  }
  
  @throw EXCEPTION(@"Patch not found in patch name list");
}

CFPropertyListRef Destination_Programs(Destination *destination, int channel, CFPropertyListRef bank)
{
  if (CFGetTypeID(bank) != CFStringGetTypeID())
    @throw EXCEPTION(@"Bank is not of CFStringRef type");

  CFArrayRef banks = (CFArrayRef) Destination_Banks(destination, channel);
  
  Boolean found = CFArrayContainsValue(banks, CFRangeMake(0, CFArrayGetCount(banks)), bank);
  
  CFRelease(banks);

  if (!found)
    @throw EXCEPTION(@"Bank is not in name set for channel of destination device");
    
  MIDIDeviceRef device = ExternalDeviceFromDestinationEndpoint(destination->endpoint);
  
  NSXMLElement *patchNameListXMLElement = PatchNameListForBankFromDevice(device, (NSString *) bank);

  NSArray *patches = PatchesFromPatchNameList(patchNameListXMLElement);
  
  /* Ownership of CFPropertyListRef return value is always passed to caller.  NSObject return values follow Objective C convention and are autoreleased.  */
  CFArrayRef patches2 = CFArrayCreateCopy(NULL, (CFArrayRef) patches);
  
  return patches2;
}

CFPropertyListRef Destination_ProgramChange(Destination *destination, int channel, CFPropertyListRef bank, CFPropertyListRef patch)
{
  if (CFGetTypeID(bank) != CFStringGetTypeID())
    @throw EXCEPTION(@"Bank is not of CFStringRef type");

  if (CFGetTypeID(patch) != CFStringGetTypeID())
    @throw EXCEPTION(@"Patch is not of CFStringRef type");

  CFArrayRef banks = (CFArrayRef) Destination_Banks(destination, channel);
  
  Boolean found = CFArrayContainsValue(banks, CFRangeMake(0, CFArrayGetCount(banks)), bank);
  
  CFRelease(banks);
  
  if (!found)
    @throw EXCEPTION(@"Bank is not in name set for channel of destination device");
  
  MIDIDeviceRef device = ExternalDeviceFromDestinationEndpoint(destination->endpoint);
  
  NSXMLElement *patchNameListXMLElement = PatchNameListForBankFromDevice(device, (NSString *) bank);
  
  NSData *programChange = ProgramChangeForPatchFromPatchNameList(patchNameListXMLElement, channel, (NSString *) patch);
  
  /* Ownership of CFPropertyListRef return value is always passed to caller.  NSObject return values follow Objective C convention and are autoreleased.  */
  CFDataRef programChange2 = CFDataCreateCopy(NULL, (CFDataRef) programChange);
  
  return programChange2;
}

static NSData *MIDIDataFromControlChangeXMLNode(NSXMLNode *controlChangeXMLNode, int channel)
{
  char MIDIControlChange[3];
  MIDIControlChange[0] = 0xb0 | ((channel - 1) & 0xf);
  MIDIControlChange[1] = 0;
  MIDIControlChange[2] = 0;
  
  NSError *err = nil;
  NSArray *controls = [controlChangeXMLNode nodesForXPath:@"./@Control" error:&err];
  if ([controls count] == 1)
  {
    NSXMLNode *control = [controls objectAtIndex:0];
    //NSLog(@"control = %@", [control objectValue]);
    MIDIControlChange[1] = [[control objectValue] intValue];
  }
  
  NSArray *values = [controlChangeXMLNode nodesForXPath:@"./@Value" error:&err];
  if ([values count] == 1)
  {
    NSXMLNode *value = [values objectAtIndex:0];
    //NSLog(@"value = %@", [value objectValue]);
    MIDIControlChange[2] = [[value objectValue] intValue];
  }
  
  return [NSData dataWithBytes:MIDIControlChange length:3];
}

static NSData *MIDIDataFromProgramChangeXMLNode(NSXMLNode *programChangeXMLNode, int channel)
{
  char MIDIProgramChange[2];
  MIDIProgramChange[0] = 0xc0 | ((channel - 1) & 0xf);
  MIDIProgramChange[1] = 0;
  
  NSError *err = nil;
  NSArray *numbers = [programChangeXMLNode nodesForXPath:@"./@Number" error:&err];
  if ([numbers count] == 1)
  {
    NSXMLNode *number = [numbers objectAtIndex:0];
    //NSLog(@"number = %@", [number objectValue]);
    MIDIProgramChange[1] = [[number objectValue] intValue];
  }
  
  return [NSData dataWithBytes:MIDIProgramChange length:2];
}

static NSData *MIDIDataFromLocalControlXMLNode(NSXMLNode *localControlXMLNode, int channel)
{
  char MIDILocalControl[3];  /* Local control is really just a control change.  */
  MIDILocalControl[0] = 0xb0 | ((channel - 1) & 0xf);
  MIDILocalControl[1] = 122;
  MIDILocalControl[2] = 0;
  
  NSError *err = nil;
  NSArray *values = [localControlXMLNode nodesForXPath:@"./@Value" error:&err];
  if ([values count] == 1)
  {
    NSXMLNode *value = [values objectAtIndex:0];
    //NSLog(@"value = %@", [value objectValue]);
    if ([[value objectValue] isEqualToString:@"on"])
      MIDILocalControl[2] = 127;
  }
  
  return [NSData dataWithBytes:MIDILocalControl length:3];
}

static NSData *MIDIDataFromSysExText(NSXMLNode *sysExText)
{
  NSMutableData *MIDIData = [NSMutableData dataWithCapacity:5];  
  
  NSArray *hexByteArray = [[[sysExText stringValue] stringByTrimmingCharactersInSet:[NSCharacterSet whitespaceCharacterSet]] componentsSeparatedByString:@" "];
  NSEnumerator *enumerator = [hexByteArray objectEnumerator];
  id hexByte;
  while (hexByte = [enumerator nextObject]) {
    unsigned int hexByteValue;
    if ([[NSScanner scannerWithString:hexByte] scanHexInt:&hexByteValue] && hexByteValue <= 255)
    {
      unsigned char value = hexByteValue & 0xff;
      [MIDIData appendBytes:&value length:1];
    }
    else
      @throw EXCEPTION(@"Invalid SysEx byte value");
  }
  
  return MIDIData;
}

static void MultiplierAndOffsetFromXMLNode(NSXMLNode *node, int *multiplierValue, int *offsetValue)
{
  *multiplierValue = 1;
  *offsetValue = 0;
  
  NSError *err = nil;
  NSArray *multipliers = [node nodesForXPath:@"./@Multiplier" error:&err];
  if ([multipliers count] == 1)
  {
    NSXMLNode *multiplier = [multipliers objectAtIndex:0];
    //NSLog(@"control = %@", [multiplier objectValue]);
    *multiplierValue = [[multiplier objectValue] intValue];
  }
  
  NSArray *offsets = [node nodesForXPath:@"./@Offset" error:&err];
  if ([offsets count] == 1)
  {
    NSXMLNode *offset = [offsets objectAtIndex:0];
    //NSLog(@"value = %@", [offset objectValue]);
    *offsetValue = [[offset objectValue] intValue];
  }
}

static NSData *MIDIDataFromSysExDeviceID(NSXMLNode *sysExDeviceID, int deviceID)
{
  int multiplierValue;
  int offsetValue;
  
  MultiplierAndOffsetFromXMLNode(sysExDeviceID, &multiplierValue, &offsetValue);
  
  char value = deviceID * multiplierValue + offsetValue;
  
  return [NSData dataWithBytes:&value length:1];
}

static NSData *MIDIDataFromSysExChannel(NSXMLNode *sysExDeviceID, int channel)
{
  int multiplierValue;
  int offsetValue;
  
  MultiplierAndOffsetFromXMLNode(sysExDeviceID, &multiplierValue, &offsetValue);
  
  char value = (channel - 1) * multiplierValue + offsetValue;
  
  return [NSData dataWithBytes:&value length:1];
}

static NSData *MIDIDataFromSysExXMLNode(NSXMLNode *sysExXMLNode, int channel, int deviceID)
{
  NSMutableData *MIDIData = [NSMutableData dataWithCapacity:25];  
  
  int count = [sysExXMLNode childCount];
  for (int i = 0; i < count; i++)
  {
    NSXMLNode *child = [sysExXMLNode childAtIndex:i];
    
    NSData *partialMIDIData;
    if ([child kind] == NSXMLTextKind)
    {
      //NSLog(@"text child %d = %@", i, [child XMLStringWithOptions:NSXMLNodePrettyPrint]);
      partialMIDIData = MIDIDataFromSysExText(child);
    }
    else if ([child kind] == NSXMLElementKind)
    {
      //NSLog(@"element child %d = %@", i, [child XMLStringWithOptions:NSXMLNodePrettyPrint]);
      //NSLog(@"name = %@", [child name]);      
      if ([[child name] isEqualToString:@"SysExDeviceID"])
	partialMIDIData = MIDIDataFromSysExDeviceID(child, deviceID);
      else if ([[child name] isEqualToString:@"SysExChannel"])
	partialMIDIData = MIDIDataFromSysExChannel(child, channel);
      else
	@throw EXCEPTION(@"Unknown element embedded in sysex");
    }
    else
    {
      NSString *exceptionMessage = [NSString stringWithFormat:@"Unknown type of element in sysex: %@", [child XMLStringWithOptions:NSXMLNodePrettyPrint]];
      @throw EXCEPTION(exceptionMessage);
    }
    
    [MIDIData appendData:partialMIDIData];
  }
  
  return MIDIData;
}

static NSData *MIDIDataFromMIDICommandsXMLNode(NSXMLNode *MIDICommandsXMLNode, int channel, int deviceID)
{
  NSMutableData *MIDIData = [NSMutableData dataWithCapacity:10];
  
  int count = [MIDICommandsXMLNode childCount];
  for (int i = 0; i < count; i++)
  {
    NSXMLNode *child = [MIDICommandsXMLNode childAtIndex:i];
    //NSLog(@"child %d = %@", i, [child XMLStringWithOptions:NSXMLNodePrettyPrint]);
    
    NSData *commandMIDIData;
    if ([[child name] isEqualToString:@"ControlChange"])
      commandMIDIData = MIDIDataFromControlChangeXMLNode(child, channel);
    else if ([[child name] isEqualToString:@"ProgramChange"])
      commandMIDIData = MIDIDataFromProgramChangeXMLNode(child, channel);
    else if ([[child name] isEqualToString:@"LocalControl"])
      commandMIDIData = MIDIDataFromLocalControlXMLNode(child, channel);
    else if ([[child name] isEqualToString:@"SysEx"])
      commandMIDIData = MIDIDataFromSysExXMLNode(child, channel, deviceID);
    else
      @throw EXCEPTION(@"Unrecognized MIDI command");
    
    [MIDIData appendData:commandMIDIData];
  }
  
  return MIDIData;
}

static int DeviceIDFromDeviceDescription(CFPropertyListRef deviceDescription)
{
  NSArray *devices = [(NSDictionary *) deviceDescription valueForKey:@"Devices"];
  
  if (!devices || [devices count] != 1)
    return 0;
  
  NSDictionary *externalDeviceDescription = [devices objectAtIndex:0];
  
  NSNumber *deviceID = [externalDeviceDescription valueForKey:@"DeviceID"];
  
  if (deviceID)
    return [deviceID intValue];
  else
    return 0;
}

static NSData *BankSelectFromXMLDocument(NSXMLDocument *xmlDoc, NSString *bank, int channel, int deviceID)
{
  NSError *err = nil;
  NSArray *MIDICommandsArray = [xmlDoc nodesForXPath:[NSString stringWithFormat:@"./MIDINameDocument/MasterDeviceNames/ChannelNameSet/PatchBank[@Name=\"%@\"]/MIDICommands", bank] error:&err];
  if (err || MIDICommandsArray == nil || [MIDICommandsArray count] != 1)
    @throw EXCEPTION(@"Cannot get bank select MIDI commands for bank");
  
  NSXMLNode *MIDICommands = [MIDICommandsArray objectAtIndex:0];
  return MIDIDataFromMIDICommandsXMLNode(MIDICommands, channel, deviceID);
}

CFPropertyListRef Destination_BankSelect(Destination *destination, int channel, CFPropertyListRef bank)
{
  if (CFGetTypeID(bank) != CFStringGetTypeID())
    @throw EXCEPTION(@"Bank is not of CFStringRef type");

  CFArrayRef banks = (CFArrayRef) Destination_Banks(destination, channel);
  
  Boolean found = CFArrayContainsValue(banks, CFRangeMake(0, CFArrayGetCount(banks)), bank);
  
  CFRelease(banks);
  
  if (!found)
    @throw EXCEPTION(@"Bank is not in name set for channel of destination device");
  
  MIDIDeviceRef device = ExternalDeviceFromDestinationEndpoint(destination->endpoint);
  
  NSXMLDocument *masterNameDocumentXMLDocument = XMLDocumentFromDevice(device);
  
  int deviceID = DeviceIDFromDeviceDescription(destination->deviceDescription);
  
  NSData *bankSelect = BankSelectFromXMLDocument(masterNameDocumentXMLDocument, (NSString *) bank, channel, deviceID);
  
  /* Ownership of CFPropertyListRef return value is always passed to caller.  NSObject return values follow Objective C convention and are autoreleased.  */
  CFDataRef bankSelect2 = CFDataCreateCopy(NULL, (CFDataRef) bankSelect);
  
  return bankSelect2;
}

