/* Copyright (c) 2005 Andrew Choi.  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.

3. All advertising materials mentioning features or use of this
   software must display the following acknowledgement:

     This product includes software developed by Andrew Choi.

4. The name "Andrew Choi" may not be used to endorse or promote
   products derived from this software without specific prior written
   permission.

THIS SOFTWARE IS PROVIDED BY ANDREW CHOI "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.  IN NO EVENT SHALL ANDREW CHOI BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.  */

#include <Foundation/Foundation.h>

#include "MIDI.h"
#include "Device.h"

#include "Destination.h"

Destination *new_Destination(CFPropertyListRef deviceDescription)
{
  /* Protect against use when client is uninitialized.  */
  if (!myMIDIClient)
    @throw EXCEPTION(@"TOE must be initialized before destination can be created");
  
  Destination *destination = CFAllocatorAllocate(NULL, sizeof(Destination), 0);
  
  CFPropertyListRef checkedDeviceDescription = CheckDeviceDescription(deviceDescription);
  
  destination->deviceDescription = checkedDeviceDescription;
  
  destination->endpoint = DestinationEndpointFromDeviceDescription(checkedDeviceDescription);

  CFStringRef displayName = CFDictionaryGetValue(checkedDeviceDescription, CFSTR("DisplayName"));
  CFStringRef portName = CFStringCreateWithFormat(NULL, NULL, CFSTR("TOE Destination Port %@"), displayName);
  
  if (MIDIOutputPortCreate(myMIDIClient, portName, &destination->port) != noErr)
  {
    CFRelease(portName);
    CFRelease(destination->deviceDescription);
    CFAllocatorDeallocate(NULL, destination);
    @throw EXCEPTION(@"Cannot create output port");
  }
  
  CFRelease(portName);
  
  return destination;
}

void delete_Destination(Destination *destination)
{
  MIDIPortDispose(destination->port);
  
  CFRelease(destination->deviceDescription);
  
  CFAllocatorDeallocate(NULL, destination);
}

CFStringRef Destination_Description(Destination *destination)
{
  SInt32 uid;
  MIDIObjectGetIntegerProperty(destination->endpoint, kMIDIPropertyUniqueID, &uid);
  
  return CFStringCreateWithFormat(NULL, NULL, CFSTR("<Destination '%@', endpoint'%d'>"), destination->deviceDescription, uid);
}

void Destination_Send(Destination *destination, CFPropertyListRef data)
{
  if (CFGetTypeID(data) != CFDataGetTypeID())
    @throw EXCEPTION(@"Data to be sent is not of CFDataRef type");
  
  CFIndex bufferSize = sizeof(UInt32) + CFDataGetLength(data) + sizeof(MIDITimeStamp) + sizeof(UInt16);
  unsigned char *buffer = CFAllocatorAllocate(NULL, bufferSize, 0);
  MIDIPacketList *packetList = (MIDIPacketList *)buffer;

  MIDIPacket *packagePointer = MIDIPacketListInit(packetList);
  
  if (!MIDIPacketListAdd(packetList, bufferSize, packagePointer, 0, CFDataGetLength(data), CFDataGetBytePtr(data)))
  {
    CFAllocatorDeallocate(NULL, buffer);    
    @throw EXCEPTION(@"Cannot add data to packet list to send");
  }
  
  OSStatus s = MIDISend(destination->port, destination->endpoint, packetList);
  
  CFAllocatorDeallocate(NULL, buffer);
  
  if (s != noErr)
    @throw EXCEPTION(@"Cannot send data to destination");
}
