%include exception.i

%{
#include <Foundation/Foundation.h>
  
CFPropertyListRef CFPropertyListFromPyObject(PyObject *py_obj)
{
  if (PyUnicode_Check(py_obj))
    return CFStringCreateWithCharacters(NULL, PyUnicode_AsUnicode(py_obj), PyUnicode_GetSize(py_obj));
  else if (PyString_Check(py_obj))
    return CFDataCreate(NULL, (UInt8 *)PyString_AsString(py_obj), PyString_Size(py_obj));
  else if (PyInt_Check(py_obj))
  {
    long l = PyInt_AsLong(py_obj);
    return CFNumberCreate(kCFAllocatorDefault, kCFNumberLongType, &l);
  }
  else if (PyFloat_Check(py_obj))
  {
    double d = PyFloat_AsDouble(py_obj);
    return CFNumberCreate(kCFAllocatorDefault, kCFNumberDoubleType, &d);
  }
  else if (PySequence_Check(py_obj))
  {
    int len = PySequence_Length(py_obj);
    
    CFPropertyListRef *values = CFAllocatorAllocate(kCFAllocatorDefault, sizeof(CFPropertyListRef) * len, 0);
    if (!values)
      return NULL;
    
    for (int i = 0; i < len; i++)
    {
      PyObject *o = PySequence_GetItem(py_obj, i);
      
      values[i] = CFPropertyListFromPyObject(o);

      if (!values[i])
      {
	for (int j = 0; j < i; j++)
	  CFRelease(values[j]);
	CFAllocatorDeallocate(kCFAllocatorDefault, values);
	return NULL;
      }
    }
    
    CFArrayRef a = CFArrayCreate(kCFAllocatorDefault, values, len, &kCFTypeArrayCallBacks);
    
    for (int i = 0; i < len; i++)
      CFRelease(values[i]);
    CFAllocatorDeallocate(kCFAllocatorDefault, values);
    
    return a;
  }
  else if (PyDict_Check(py_obj))
  {
    int size = PyDict_Size(py_obj);
    
    CFPropertyListRef *keys = CFAllocatorAllocate(kCFAllocatorDefault, sizeof(CFPropertyListRef) * size, 0);
    if (!keys)
      return NULL;    
    CFPropertyListRef *values = CFAllocatorAllocate(kCFAllocatorDefault, sizeof(CFPropertyListRef) * size, 0);
    if (!values)
    {
      CFAllocatorDeallocate(kCFAllocatorDefault, keys);
      return NULL;
    }
    
    int i = 0;  /* Not clear if 'pos' is always incremented by one.  */
    PyObject *key, *value;
    int pos = 0;
    while (PyDict_Next(py_obj, &pos, &key, &value))
    {
      if (!PyUnicode_Check(key))
      {
	for (int j = 0; j < i; j++)
	{
	  CFRelease(keys[j]);
	  CFRelease(values[j]);
	}
	
	CFAllocatorDeallocate(kCFAllocatorDefault, keys);
	CFAllocatorDeallocate(kCFAllocatorDefault, values);
	
	return NULL;
      }
      
      keys[i] = CFPropertyListFromPyObject(key);
      if (!keys[i])
      {
	for (int j = 0; j < i; j++)
	{
	  CFRelease(keys[j]);
	  CFRelease(values[j]);
	}
	
	CFAllocatorDeallocate(kCFAllocatorDefault, keys);
	CFAllocatorDeallocate(kCFAllocatorDefault, values);

	return NULL;
      }
      
      values[i] = CFPropertyListFromPyObject(value);
      if (!values[i])
      {
	CFRelease(keys[i]);
	
	for (int j = 0; j < i; j++)
	{
	  CFRelease(keys[j]);
	  CFRelease(values[j]);
	}

	CFAllocatorDeallocate(kCFAllocatorDefault, keys);
	CFAllocatorDeallocate(kCFAllocatorDefault, values);
	
	return NULL;
      }
      
      i++;
    }
    
    CFDictionaryRef d = CFDictionaryCreate(kCFAllocatorDefault, keys, values, size, &kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks);
    
    for (int i = 0; i < size; i++)
    {
      CFRelease(keys[i]);
      CFRelease(values[i]);
    }
    CFAllocatorDeallocate(kCFAllocatorDefault, keys);
    CFAllocatorDeallocate(kCFAllocatorDefault, values);
    
    return d;
  }
  else
    return NULL;
}

PyObject *PyObjectFromCFPropertyList(CFPropertyListRef plist)
{
  CFTypeID type = CFGetTypeID(plist);
  if (type  == CFStringGetTypeID())
  {
    CFIndex len = CFStringGetLength(plist);
    const UniChar *buf = CFStringGetCharactersPtr(plist);
    if (buf)
      return PyUnicode_FromUnicode(buf, len);
    else
    {
      UniChar *buf2 = CFAllocatorAllocate(kCFAllocatorDefault, len * sizeof(UniChar), 0);
      CFStringGetCharacters(plist, CFRangeMake(0, len), buf2);
      PyObject *result = PyUnicode_FromUnicode(buf2, len);
      
      CFAllocatorDeallocate(kCFAllocatorDefault, buf2);
      
      return result;
    }
  }
  else if (type == CFDataGetTypeID())
    return PyString_FromStringAndSize((const char *)CFDataGetBytePtr(plist), CFDataGetLength(plist));
  else if (type == CFNumberGetTypeID())
  {
    if (CFNumberIsFloatType(plist))
    {
      double d;
      if (CFNumberGetValue(plist, kCFNumberDoubleType, &d))
	return PyFloat_FromDouble(d);
      else
	return NULL;
    }
    else
    {
      long l;
      if (CFNumberGetValue(plist, kCFNumberLongType, &l))
	return PyInt_FromLong(l);
      else
	return NULL;
    }
  }
  else if (type == CFArrayGetTypeID())
  {
    CFIndex count = CFArrayGetCount(plist);
    PyObject *l = PyList_New(count);
    
    for (int i = 0; i < count; i++)
    {
      CFPropertyListRef pl_item = CFArrayGetValueAtIndex(plist, i);
      
      PyObject *py_item = PyObjectFromCFPropertyList(pl_item);
      
      if (!py_item)
      {
	Py_DECREF(l);
	return NULL;
      }
      
      PyList_SetItem(l, i, py_item);
    }
    
    return l;
  }
  else if (type == CFDictionaryGetTypeID())
  {
    CFIndex count = CFDictionaryGetCount(plist);
    
    CFPropertyListRef *keys = CFAllocatorAllocate(NULL, sizeof(CFPropertyListRef) * count, 0);
    if (!keys)
      return NULL;    
    CFPropertyListRef *values = CFAllocatorAllocate(NULL, sizeof(CFPropertyListRef) * count, 0);
    if (!values)
    {
      CFAllocatorDeallocate(NULL, keys);
      return NULL;
    }
    
    CFDictionaryGetKeysAndValues(plist, keys, values);
    
    PyObject *d = PyDict_New();
    
    for (int i = 0; i < count; i++)
    {
      PyObject *py_key = PyObjectFromCFPropertyList(keys[i]);
      
      if (!py_key)
      {
	Py_DECREF(d);

	CFAllocatorDeallocate(kCFAllocatorDefault, keys);
	CFAllocatorDeallocate(kCFAllocatorDefault, values);
	
	return NULL;
      }

      PyObject *py_value = PyObjectFromCFPropertyList(values[i]);
      if (!py_value)
      {
	Py_DECREF(d);
	Py_DECREF(py_key);

	CFAllocatorDeallocate(kCFAllocatorDefault, keys);
	CFAllocatorDeallocate(kCFAllocatorDefault, values);

	return NULL;
      }
      
      PyDict_SetItem(d, py_key, py_value);
      
      Py_DECREF(py_key);
      Py_DECREF(py_value);
    }

    CFAllocatorDeallocate(kCFAllocatorDefault, keys);
    CFAllocatorDeallocate(kCFAllocatorDefault, values);

    return d;
  }
  else
    return NULL;
}

%}

%typemap(in) CFPropertyListRef
{
  $1 = CFPropertyListFromPyObject($input);
  
  if (!$1)
    SWIG_exception(SWIG_ValueError, "Cannot convert parameter to plist");
}

%typemap(freearg) CFPropertyListRef
{
  if ($1)
    CFRelease($1);
}

%typemap(arginit) CFPropertyListRef
{
  $1 = NULL;
}

%typemap(out) CFPropertyListRef
{
  $result = PyObjectFromCFPropertyList($1);

  if ($1)
    CFRelease($1);

  if (!$result)
    SWIG_exception(SWIG_ValueError, "Can't convert plist to result");
}

%exception {
  Boolean exception_occurred = false;
  char *exception_message = NULL;
  
  NSAutoreleasePool * pool = [[NSAutoreleasePool alloc] init];
  
  @try {
    $action
  }
  @catch(NSException *exception) {
    exception_occurred = true;
    
    const char * utf8_exception_message = [[NSString stringWithFormat:@"%@ [in function %@]", [exception reason], [exception name]] UTF8String];
    exception_message = alloca(strlen(utf8_exception_message) + 1);
    strcpy(exception_message, utf8_exception_message);
  }
  
  [pool release];
  
  if (exception_occurred)
    SWIG_exception(SWIG_RuntimeError, exception_message);
}
