/*
  Permission for the use of this code is granted only for research, educational, and non-commercial purposes.

  Redistribution of this code or its parts in source, binary, and any other form without permission, with or without modification, is prohibited.  Modifications include, but are not limited to, translation to other programming languages and reuse of tables, constant definitions, and API's defined in it.

  Andrew Choi is not liable for any losses or damages caused by the use of this code.

  Copyright 2009 Andrew Choi.
*/

#include <QtGui>
#include <QtDebug>

#include "documentwindowmanager.h"

#include "qtdocbasedapplication.h"

// The QtApplication virtual function commitData doesn't seem to be called on Mac.  So a way to handle an external request to terminate this application seems to be to catch the Apple event kAEQuitApplication.  A little Carbon code.  No big deal.

#ifdef Q_WS_MAC
const QEvent::Type kMacQuitEventType = QEvent::Type(QEvent::registerEventType());
#endif

QtDocBasedApplication::QtDocBasedApplication(int &argc, char **argv) : QtSingleApplication(argc, argv)
{
#ifdef Q_WS_MAC
    setQuitOnLastWindowClosed(false);

#ifndef QT_MAC_USE_COCOA
    AEEventHandlerUPP qaHandler = NewAEEventHandlerUPP(quitApplicationAeHandler);

    // This overrides AE handler installed by Qt (src/gui/kernel/qapplication_mac.mm).
    if (AEInstallEventHandler(kCoreEventClass, kAEQuitApplication, qaHandler, 0L, false) != noErr)
        qFatal("Cannot install quit application Apple event handler");

    DisposeAEEventHandlerUPP(qaHandler);
#endif
#endif
}

QtDocBasedApplication::~QtDocBasedApplication()
{
#ifdef Q_WS_MAC
#ifndef QT_MAC_USE_COCOA
    AERemoveEventHandler(kCoreEventClass, kAEQuitApplication, 0, false);
#endif
#endif
}

#ifdef Q_WS_MAC
// This opens a document using this application from the Finder.
bool QtDocBasedApplication::event(QEvent *event)
{
    if (event->type() == QEvent::FileOpen)
    {
        DocumentWindowManager::instance()->openFile(static_cast<QFileOpenEvent *>(event)->file());

        return true;
    }

    return QApplication::event(event);
}
#endif

// Does not seem to be called on Mac.
#ifndef Q_WS_MAC
void QtDocBasedApplication::commitData(QSessionManager &sm)
{
    int modifiedDocCount = DocumentWindowManager::instance()->countModifiedDocs();

    if (modifiedDocCount == 0)
        return;

    if (sm.allowsInteraction())
    {
        QMessageBox logoutMessageBox(0);

        logoutMessageBox.setIcon(QMessageBox::Warning);
        logoutMessageBox.setText(tr("You have %1 QtDocBasedApp documents with unsaved changes. Do you want to cancel logout so that you can have a chance to first save them?").arg(modifiedDocCount));
        logoutMessageBox.setInformativeText(tr("If you click “Discard Changes and Logout”, all your changes will be lost."));
        logoutMessageBox.setDefaultButton(logoutMessageBox.addButton(tr("Cancel Logout"), QMessageBox::AcceptRole));
        logoutMessageBox.addButton(tr("Discard Changes and Logout"), QMessageBox::DestructiveRole);

        logoutMessageBox.exec();

        switch(logoutMessageBox.buttonRole(logoutMessageBox.clickedButton()))
        {
        case QMessageBox::DestructiveRole:
            sm.release();
            DocumentWindowManager::instance()->forceCloseAllDocs();
            break;
        case QMessageBox::AcceptRole:
        default:
            sm.cancel();
        }
    }
}
#endif

#ifdef Q_WS_MAC
#ifndef QT_MAC_USE_COCOA
OSErr QtDocBasedApplication::quitApplicationAeHandler(const AppleEvent *theAppleEvent, AppleEvent *, long)
{
    DescType returnedType;
    ProcessSerialNumber senderPSN;
    Size actualSize;

    if (AEGetAttributePtr(theAppleEvent, keyOriginalAddressAttr, typeProcessSerialNumber, &returnedType, &senderPSN, sizeof(ProcessSerialNumber), &actualSize) == noErr)
    {
        if (senderPSN.highLongOfPSN == 0 && senderPSN.lowLongOfPSN == 1)
        {
            // PSN of loginwindow = (0, 1), which means the incoming kAEQuitApplication AE results from logout, shutdown or restart.

            int modifiedDocCount = DocumentWindowManager::instance()->countModifiedDocs();

            if (modifiedDocCount == 0)
            {
                DocumentWindowManager::instance()->closeDocumentsAndQuit();

                return noErr;
            }

            // Resorting to a Carbon dialog here because this event handler must return a userCanceledErr value to cancel the logout/shutdown/restart.  The "event-driven" design of Qt prevents us from using its dialogs and still getting back here.
            CFStringRef s1 = CFStringCreateWithFormat(NULL, NULL, CFSTR("You have %d QtDocBasedApp documents with unsaved changes. Do you want to cancel logout so that you can have a chance to first save them?"), modifiedDocCount);
            CFStringRef s2 = CFStringCreateWithFormat(NULL, NULL, CFSTR("If you click %CDiscard Changes and Logout%C, all your changes will be lost."), 0x201c, 0x201d);

            AlertStdCFStringAlertParamRec paramRec;

            GetStandardAlertDefaultParams(&paramRec, kStdCFStringAlertVersionOne);
            paramRec.defaultText = CFSTR("Cancel Logout");
            paramRec.defaultButton = kAlertStdAlertOKButton;
            paramRec.cancelText = CFSTR("Discard Changes and Logout");
            paramRec.cancelButton = kAlertStdAlertCancelButton;

            DialogRef dialogRef;
            CreateStandardAlert(kAlertStopAlert, s1, s2, &paramRec, &dialogRef);

            DialogItemIndex dialogItemIndex;
            RunStandardAlert(dialogRef, NULL, &dialogItemIndex);

            if (dialogItemIndex == kAlertStdAlertOKButton)
                return userCanceledErr;
            else
            {
                DocumentWindowManager::instance()->forceCloseAllDocs();
                qApp->quit();

                return noErr;
            }
        }
        else
        {
            // Otherwise the kAEQuitApplication AE results from selecting Quit in the application icon menu in the dock.
            QEvent macQuitEvent = QEvent(kMacQuitEventType);  // custom event macQuitEvent, which will be picked up by the event handler customEvent(QEvent) below

            QApplication::sendEvent(qApp, &macQuitEvent);

            return noErr;
        }
    }
    else
    {
        qFatal("Internal error: cannot get sender PSN from kAEQuitApplication Apple event");

        return noErr;
    }
}

void QtDocBasedApplication::customEvent(QEvent *event)
{
    if (event->type() == kMacQuitEventType)
        DocumentWindowManager::instance()->closeDocumentsAndQuit();
}

#endif
#endif

