/*
  Permission for the use of this code is granted only for research, educational, and non-commercial purposes.

  Redistribution of this code or its parts in source, binary, and any other form without permission, with or without modification, is prohibited.  Modifications include, but are not limited to, translation to other programming languages and reuse of tables, constant definitions, and API's defined in it.

  Andrew Choi is not liable for any losses or damages caused by the use of this code.

  Copyright 2009 Andrew Choi.
*/

#include <QtGui>
#include <QtDebug>

#include "chordvalidator.h"
#include "chordnamemodel.h"

#include "chordedit.h"

ChordEdit::ChordEdit()
{
    // Provide sizeHint below and call updateGeometry in handleTextChange to keep widget just big enough to contain chord.
    setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);

    ChordValidator *validator = new ChordValidator(this);
    setValidator(validator);

    model = new ChordNameModel(this);
    completer = new QCompleter(model, this);  // Create a completer but only invoke it manually in keyPressEvent.
    completer->setCompletionMode(QCompleter::PopupCompletion);
    completer->setCaseSensitivity(Qt::CaseSensitive);
    completer->setModelSorting(QCompleter::UnsortedModel);
    completer->setWrapAround(false);
    completer->setWidget(this);

    connect(completer, SIGNAL(activated(const QString &)), this, SLOT(setTextAndHide(const QString &)));
    connect(completer, SIGNAL(highlighted(const QString &)), this, SLOT(setTextAndHighlight(const QString &)));

    setFont(qApp->font());

    // Hook into our own signal: call handleTextChange when text has changed.
    connect(this, SIGNAL(textChanged(const QString &)), this, SLOT(handleTextChange(const QString &)));
}

// Each ChordEdit must use a single font (since the base class QLineEdit does).  However the codes for the # and b characters may differ.  By default use the Unicode # (0x266f) and b (0x266d).
void ChordEdit::setFont(const QFont &f)
{
    QString s = decode(text());

    QLineEdit::setFont(f);
    completer->popup()->setFont(f);

    if (f.family() == "JazzText")
    {
        displaySharpSymbol = QChar('#');
        displayFlatSymbol = QChar(0xd1);
    }
    else if (f.family() == "MaestroTimes")
    {
        displaySharpSymbol = QChar(0x2265);
        displayFlatSymbol = QChar(0x2264);
    }
    else
    {
        displaySharpSymbol = QChar(0x266f);
        displayFlatSymbol = QChar(0x266d);
    }

    setText(encode(s));
}

// See QLineEdit::sizeHint()
const int verticalMargin = 1;
const int horizontalMargin = 2;

QSize ChordEdit::sizeHint() const
{
    ensurePolished();

    int ml, mt, mr, mb;
    getContentsMargins(&ml, &mt, &mr, &mb);

    QFontMetrics fm(font());
    int h = qMax(fm.lineSpacing(), 14) + 2 * verticalMargin + mt + mb;
    int w = qMax(fm.width(text()), fm.width(QLatin1Char('x')) * 4) + 2 * horizontalMargin + ml + mr;
#ifndef Q_WS_WIN
    QStyleOptionFrameV2 opt;
    initStyleOption(&opt);
    return (style()->sizeFromContents(QStyle::CT_LineEdit, &opt, QSize(w, h).expandedTo(QApplication::globalStrut()), this));
#else
    // On Windows, sizeFromContents returns a box that's way too wide.
    return QSize(w + 4, h + 2);
#endif
}

// Sharp and flat symbols used for display are different than those used internally in chordTypeNames: # and b (letter B).  Encode and decode convert between the two encodings.
QString ChordEdit::encode(const QString &str)
{
    QString result(str);

    return result.replace("#", displaySharpSymbol).replace("b", displayFlatSymbol);
}

QString ChordEdit::decode(const QString &str)
{
    QString result(str);

    return result.replace(displaySharpSymbol, "#").replace(displayFlatSymbol, "b");
}

void ChordEdit::handleTextChange(const QString &s)
{
    Q_UNUSED(s);

    updateGeometry();

    // Change prefix for completer if the user types a key that is not intercepted by the completion popup (i.e., not up/down arrow etc.).
    if (keyPressedDuringCompletion && !completer->popup()->isHidden())
    {
        completer->setCompletionPrefix(s);
        completionPrefixLength = s.length();

        keyPressedDuringCompletion = false;

        // Adjust size of popup.
        QSize bps = bestPopupSize();
        completer->popup()->setMinimumSize(bps);  // Seems to need both setMinimumSize and resize.
        completer->popup()->resize(bps);
    }
}

void ChordEdit::setTextAndHide(const QString &s)
{
    setText(s);

    completer->popup()->hide();
}

// Mimic Mac behavior by putting selected completion in text box and highlighting the portin that's not the prefix.
void ChordEdit::setTextAndHighlight(const QString &s)
{
    setText(s);

    setSelection(completionPrefixLength, s.length() - completionPrefixLength);
}

void ChordEdit::keyPressEvent(QKeyEvent *e)
{
    if (!completer->popup()->isHidden())
        keyPressedDuringCompletion = true;

    if (e->key() == Qt::Key_Tab)
    {
        qDebug() << "Bing!!";
    }

    // Same as on Mac: Opt-Esc or F5 starts completion.
    if ((e->modifiers() & Qt::AltModifier && e->key() == Qt::Key_Escape) || e->key() == Qt::Key_F5)
    {
        completer->setCompletionPrefix(text());
        completionPrefixLength = text().length();

        keyPressedDuringCompletion = false;

        completer->popup()->setCurrentIndex(completer->completionModel()->index(0, 0));

        QSize bps = bestPopupSize();

        completer->popup()->setMinimumSize(bps);

        completer->complete(QRect(0, 0, bps.width(), height()));  // widget relative coordinates

        e->accept();
    }
    else
    {
        QLineEdit::keyPressEvent(e);
    }
}

const int MaxRowsInPopup = 10;

// Without this the n rows won't be displayed completely in the popup.
#if defined(Q_WS_MAC)
const int heightAdjustment = 0;
#elif defined(Q_WS_WIN)
const int heightAdjustment = 4;
#else
const int heightAdjustment = 2;
#endif

QSize ChordEdit::bestPopupSize()
{
    int w = completer->popup()->sizeHintForColumn(0) + completer->popup()->verticalScrollBar()->sizeHint().width();

    int n = completer->popup()->model()->rowCount();
    if (n > MaxRowsInPopup)
        n = MaxRowsInPopup;

    int h = (completer->popup()->sizeHintForRow(0) * n + heightAdjustment);

    return QSize(w, h);
}

