/*
  Permission for the use of this code is granted only for research, educational, and non-commercial purposes.

  Redistribution of this code or its parts in source, binary, and any other form without permission, with or without modification, is prohibited.  Modifications include, but are not limited to, translation to other programming languages and reuse of tables, constant definitions, and API's defined in it.

  Andrew Choi is not liable for any losses or damages caused by the use of this code.

  Copyright 2009 Andrew Choi.
*/

#include <QtDebug>
#include <QFile>
#include <QTextStream>

#include "constants.h"

#include "chordvalidator.h"

ChordValidator::ChordValidator(QObject *parent) : QValidator(parent)
{
}

QValidator::State ChordValidator::validate(QString &input, int &pos) const
{
    Q_UNUSED(pos);

    if (input.length() == 0)
        return QValidator::Intermediate;

    ChordEdit *chordEdit = getChordEdit();

    QString note, alt, chordType;
    if (splitChord(chordEdit->decode(input), note, alt, chordType))
    {
        QValidator::State state = caseSensitiveMatch(chordType);
        if (state != QValidator::Invalid)
        {
            input = note.toUpper() + chordEdit->encode(alt + chordType);
            return state;
        }

        // If the case-sensitive match didn't work, try a case-insensitive one.  The argument match gets back the match in the original case.  So the user can type "ma" and that gets changed to "Ma" (for Maj7, etc.) automatically.
        QString match;
        state = caseInsensitiveMatch(chordType, match);
        if (state != QValidator::Invalid)
        {
            input = note.toUpper() + chordEdit->encode(alt + match);
            return state;
        }
    }

    return QValidator::Invalid;
}

void ChordValidator::fixup(QString &input) const
{
    ChordEdit *chordEdit = getChordEdit();

    QString note, alt, chordType;
    if (splitChord(chordEdit->decode(input), note, alt, chordType))
    {
        QStringList chordTypeNames = Constants::instance()->chordTypeNames();

        for (int i = 0; i < chordTypeNames.count(); i++)
        {
            QString chordTypeName = chordTypeNames.at(i);
            if (chordTypeName.indexOf(chordType, 0, Qt::CaseInsensitive) == 0)
            {
                input = note.toUpper() + chordEdit->encode(alt + chordTypeName);
                break;
            }
        }
    }
}

ChordEdit *ChordValidator::getChordEdit() const
{
    ChordEdit *chordEdit = qobject_cast<ChordEdit *>(QObject::parent());
    if (!chordEdit)
       qFatal("Parent of ChordValidator isn't ChordEdit");

    return chordEdit;
}

bool ChordValidator::splitChord(const QString &chord, QString &note, QString &alt, QString &chordType) const
{
    QRegExp chordRegExp("([A-Ga-g])((b|bb||#|##)?)(.*)");

    if (chordRegExp.exactMatch(chord))
    {
        note = chordRegExp.cap(1);
        alt = chordRegExp.cap(2);
        chordType = chordRegExp.cap(4);

        return true;
    }
    else
        return false;
}

QValidator::State ChordValidator::caseSensitiveMatch(const QString &s) const
{
    QStringList chordTypeNames = Constants::instance()->chordTypeNames();

    QValidator::State result = QValidator::Invalid;

    for (int i = 0; i < chordTypeNames.count(); i++)
    {
        QString chordTypeName = chordTypeNames.at(i);
        if (chordTypeName == s)
        {
            return QValidator::Acceptable;
        }
        else if (chordTypeName.indexOf(s) == 0)
        {
            // Must go through entire list in case exact match comes later than intermediate one (which is the case when list is sorted).
            result = QValidator::Intermediate;
        }
    }

    return result;
}

QValidator::State ChordValidator::caseInsensitiveMatch(const QString &s, QString &match) const
{
    QStringList chordTypeNames = Constants::instance()->chordTypeNames();

    QValidator::State result = QValidator::Invalid;

    for (int i = 0; i < chordTypeNames.count(); i++)
    {
        QString chordTypeName = chordTypeNames.at(i);
        if (QString::compare(chordTypeName, s, Qt::CaseInsensitive) == 0)
        {
            match = chordTypeName;
            return QValidator::Acceptable;
        }
        else if (chordTypeName.indexOf(s, 0, Qt::CaseInsensitive) == 0)
        {
            // Go through entire list in case exact match comes later than intermediate.
            match = chordTypeName.left(s.length());
            result = QValidator::Intermediate;
        }
    }

    return result;
}
