#! /usr/bin/env python

# Copyright (c) 2006 Andrew Choi.  All rights reserved.
#
# This is emphatically NOT free/GPL software.
#
# The use of the source or any binary form of this software, with or
# without modifications, is permitted only for research, educational,
# and non-commercial purposes.
#
# Redistribution of this software or any of its parts, in source or
# binary form, with or without modification, is prohibited.
#
# There is no restriction on the use of the output generated by this
# software.
#
# Filename: lychords.py
#
# The Lilypond chord type specs, like the TOE chord type specs, are
# able to represent *any* interval lists, and thus any chords.
# Therefore translating from a TOE chord type to its Lilypond
# counterpart is an algorithmic process and not a table lookup one.

__all__ = ['toe_chord_type_to_ly']

import re

from toe.core.objects import ChordType

INTERVAL_RE = re.compile('((##|#|bb|b)?)(10|11|12|13|[1-9])')

first_interval_map = {
    'b2': '2-',
    '2': '2',
    '#2': '2+',
    'b3': '3-',
    '3': '3',
    '#3': '3+',
    'b4': 'sus4-',
    '4': 'sus4',
    '#4': 'sus4+',
    'b5': '5-^3',
    '5': '5^3',
    '#5': '5+^3' }

def translate_other_interval(s):
    # Handle special cases: Lilypond '7' and '7+' are TOE 'b7' and
    # '7', respectively.
    if s == 'b7':
        return '7'
    if s == '7':
        return '7+'
    if s == 'bb7':
        return '7-'
    
    _toe_interval_to_ly = {'b': '-', '': '', '#': '+'}
    
    _m = INTERVAL_RE.match(s)
    if not _m or _m.group(0) != s:
        raise ValueError('Invalid interval: %s' % s)
    interval = _m.group(3)
    alt = _toe_interval_to_ly.get(_m.group(1))

    if alt == None:
        raise ValueError('Interval not representable in Lilypond: %s' % s)
    
    return interval + alt

def toe_chord_type_to_ly(chord_type_name):
    chord_type = ChordType(chord_type_name)
    interval_list = chord_type._interval_list_string()[1:].split('.')

    if not interval_list:
        raise ValueError('Empty interval list for chord type: %s' % chord_type_name)
    
    ly_chord_type = ':' + first_interval_map.get(interval_list[0])
    if not ly_chord_type:
        raise ValueError('Unrecognized first interval: %s', interval_list[0])

    for interval in interval_list[1:]:
        ly_chord_type += '.' + translate_other_interval(interval)

    return ly_chord_type
