#! /usr/bin/env python

# Copyright (c) 2006 Andrew Choi.  All rights reserved.
#
# This is emphatically NOT free/GPL software.
#
# The use of the source or any binary form of this software, with or
# without modifications, is permitted only for research, educational,
# and non-commercial purposes.
#
# Redistribution of this software or any of its parts, in source or
# binary form, with or without modification, is prohibited.
#
# There is no restriction on the use of the output generated by this
# software.
#
# Filename: xmlchords.py
#
# The MusicXML chord type specs, like the TOE chord type specs, are
# able to represent *any* interval lists, and thus any chords.
# Therefore translating from a TOE chord type to its MusicXML
# counterpart and back is an algorithmic process and not a table
# lookup one.

__all__ = ['toe_chord_type_to_xml', 'xml_chord_type_to_toe']

from sets import Set
import re
from sys import maxint

from toe.core.objects import ChordType

INTERVAL_RE = re.compile('((##|#|bb|b)?)(10|11|12|13|[1-9])')

kind_intervals = [
    # Triads
    ('major', ['3', '5']),
    ('minor', ['b3', '5']),
    ('augmented', ['3', '#5']),
    ('diminished', ['b3', 'b5']),
    # Sevenths
    ('dominant', ['3', '5', 'b7']),
    ('major-seventh', ['3', '5', '7']),
    ('minor-seventh', ['b3', '5', 'b7']),
    ('diminished-seventh', ['b3', 'b5', 'bb7']),
    ('augmented-seventh', ['3', '#5', 'b7']),
    ('half-diminished', ['b3', 'b5', 'b7']),
    ('major-minor', ['b3', '5', '7']),
    # Sixths
    ('major-sixth', ['3', '5', '6']),
    ('minor-sixth', ['b3', '5', '6']),
    # Ninths
    ('dominant-ninth', ['3', '5', 'b7', '9']),
    ('major-ninth', ['3', '5', '7', '9']),
    ('minor-ninth', ['b3', '5', 'b7', '9']),  # <<< != MusicXML specs!
    # 11ths (usually as the basis for alteration)
    ('dominant-11th', ['3', '5', 'b7', '9', '11']),
    ('major-11th', ['3', '5', '7', '9', '11']),

    ('minor-11th', ['b3', '5', 'b7', '9', '11']),  # <<< ditto
    # 13ths (usually as the basis for alteration)
    ('dominant-13th', ['3', '5', 'b7', '9', '11', '13']),
    ('major-13th', ['3', '5', '7', '9', '11', '13']),
    ('minor-13th', ['b3', '5', 'b7', '9', '11', '13']),  # <<< ditto
    # Suspended
    ('suspended-second', ['2', '5']),
    ('suspended-fourth', ['4', '5']),
    # Functional sixths
    ('Neapolitan', None),
    ('Italian', None),
    ('French', None),
    ('German', None),
    # Other
    ('pedal', []),
    ('Tristan', None) ]

def type_and_alt_from_interval(interval):
    _m = INTERVAL_RE.match(interval)
    if not _m or _m.group(0) != interval:
        raise ValueError('Invalid interval: %s' % interval)
    return _m.group(3), _m.group(1)

kind_interval_set_interval_dict = []

for kind, intervals in kind_intervals:
    if intervals != None:
        interval_dict = {}
        for interval in intervals:
            interval_type, interval_alt = type_and_alt_from_interval(interval)
            interval_dict[interval_type] = interval_alt
        
        kind_interval_set_interval_dict.append((kind, Set(intervals), interval_dict))


alt_code_dict = {'': 0, '#': 1, '##': 2, 'b': -1, 'bb': -2}

def alt_code(s):
    code = alt_code_dict.get(s)
    if code == None:
        raise ValueError('Unknown degree alter spec: %s' % s)
    
    return code

def chord_difference(s1, s2, d2):
    diff1 = s1.difference(s2)
    diff2 = s2.difference(s1)

    degrees = []
    for interval in diff1:
        interval_type, interval_alt = type_and_alt_from_interval(interval)
        old_interval_alt = d2.get(interval_type)
        if old_interval_alt != None:
            degrees.append((int(interval_type), alt_code(interval_alt), 'alter'))
            diff2.remove(old_interval_alt + interval_type)
        else:
            degrees.append((int(interval_type), alt_code(interval_alt), 'add'))

    for interval in diff2:
        interval_type, interval_alt = type_and_alt_from_interval(interval)
        degrees.append((int(interval_type), alt_code(interval_alt), 'subtract'))

    return degrees
    
def difference_cost(degrees):
    cost = 0
    for degree_value, degree_alter, degree_type in degrees:
        if degree_type == 'alter':
            cost += 1
        else:
            cost += 2
    return cost

def toe_chord_type_to_xml(chord_type_name):
    chord_type = ChordType(chord_type_name)
    interval_set = Set(chord_type._interval_list_string()[1:].split('.'))

    if len(interval_set) == 0:
        raise ValueError('No interval for chord type: %s' % chord_type_name)

    best_cost = maxint
    for kind, interval_set2, interval_dict2 in kind_interval_set_interval_dict:
        degrees = chord_difference(interval_set, interval_set2, interval_dict2)
        cost = difference_cost(degrees)
        if cost < best_cost:
            best_cost = cost
            best_kind = kind
            best_degrees = degrees
            
    return best_kind, best_degrees


kind_interval_dict_map = {}

for kind, intervals in kind_intervals:
    if intervals != None:
        interval_dict = {}
        for interval in intervals:
            interval_type, interval_alt = type_and_alt_from_interval(interval)
            interval_dict[int(interval_type)] = alt_code(interval_alt)
        kind_interval_dict_map[kind] = interval_dict

alt_spec_dict = {0: '', 1: '#', 2: '##', -1: 'b', -2: 'bb'}

def alt_spec(c):
    spec = alt_spec_dict.get(c)
    if spec == None:
        raise ValueError('Unknown degree alter code: %s' % c)
        
    return spec

def xml_chord_type_to_toe(kind, degrees):
    interval_dict = kind_interval_dict_map.get(kind)
    if interval_dict == None:
        raise ValueError('Unrecognized harmony kind: %s' % kind)
    
    my_interval_dict = interval_dict.copy()

    for degree_value, degree_alter, degree_type in degrees:
        if degree_type == 'subtract':
            del my_interval_dict[degree_value]
        else:
            my_interval_dict[degree_value] = degree_alter

    interval_list = []
    interval_types = my_interval_dict.keys()
    interval_types.sort()
    #print interval_types
    for it in interval_types:
        interval_list.append(alt_spec(my_interval_dict[it]) + str(it))
    #print interval_list

    return ':' + '.'.join(interval_list)
