# Copyright (c) 2007 Andrew Choi.  All rights reserved.

# This is emphatically NOT free/GPL software.

# Permission for the use of this code is granted only for research,
# educational, and non-commercial purposes.

# Redistribution of this code or its parts in any form without
# permission, with or without modification, is prohibited.
# Modifications include, but are not limited to, translation to other
# programming languages and reuse of tables, constant definitions, and
# API's defined in it.

# There is no restriction on the use of the output generated by this
# software.

# THIS SOFTWARE IS PROVIDED BY ANDREW CHOI "AS IS" AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED.  IN NO EVENT SHALL ANDREW CHOI BE LIABLE FOR ANY DIRECT,
# INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
#  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
# STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
# IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.


from toe.midi.objects import MIDINote, Sequence

from mjb.utils import LookaheadIterator, are_approximately_equal, fractional_part
from mjb.objects import Pattern
from mjb.generators import PG

rc_pg = PG([('|x.xxx.xx|', 70),
            ('|xxx.x.xx|', 10),
            ('|x.xxxxx.|', 10),
            ('|x.x.xxx.|', 10)])

hh_pg = PG([('|x...x...|', 80),
            ('|x...x..x|', 20)])

sd_pg = PG([('|........|', 20),
            ('|......x.|', 20),
            ('|.....x..|', 20),
            ('|..x....x|', 20),
            ('|.x....x.|', 20)])

bd_pg = PG([('|x...x...|', 20),
            ('|x...x..x|', 20),
            ('|x.......|', 20),
            ('|....x...|', 20),
            ('|....x..x|', 20)])

pg = {'rc': rc_pg, 'hh': hh_pg, 'sd': sd_pg, 'bd': bd_pg}

rc_fill_pg = PG([('|x.x.x...|', 50),
                 ('|x.x.....|', 50)])

sd_fill_pg = PG([('|..x.xxxx|', 50),
                 ('|..xx.x.x|', 50)])

fill_pg = {'rc': rc_fill_pg, 'hh': hh_pg, 'sd': sd_fill_pg, 'bd': bd_pg}

ending_pattern = {'rc': Pattern('|....x...|'),
                  'hh': Pattern('|........|'),
                  'sd': Pattern('|x..xx...........|'),
                  'bd': Pattern('|...xx...|')}

instruments = ['rc', 'hh', 'sd', 'bd']

drum_map = {'rc': MIDINote(51),
            'hh': MIDINote(42),
            'sd': MIDINote(38),
            'bd': MIDINote(36)}

from drums import drum_map

velocity_map = {'rc': 100,
                'hh': 100,
                'sd': 80,
                'bd': 100}

midinote_velocity_map = dict()
for i in velocity_map.keys():
    midinote_velocity_map[drum_map[i]] = velocity_map[i]

HH_WEAK_VELOCITY_FACTOR = 0.75
RC_WEAK_VELOCITY_FACTOR = 0.7

def midify_pat_seq(seq, midinote):
    midi_seq = Sequence()
    for time, event in seq:
        pattern = event
        for beat, duration in pattern.beats:
            f = 4.0 / pattern.div_per_bar
            midi_seq.add_event(time + beat * f, (midinote, duration * f))
    return midi_seq

def perf_seq(seq):
    result = Sequence()

    for time, event in seq:
        mn, dur = event

        f = 1.0

        if are_approximately_equal(fractional_part(time), 0.5):
            if mn == drum_map['hh']:
                f = HH_WEAK_VELOCITY_FACTOR
            elif mn == drum_map['rc']:
                f = RC_WEAK_VELOCITY_FACTOR

        result.add_event(time, (mn, dur, int(midinote_velocity_map[mn] * f)))

    return result

def gen_drums(chart):
    sections = chart['sections']
    chords = chart['chords']

    drum_track = dict()
    for i in instruments:
        drum_track[i] = Sequence()

    beat = 0.0
    for bar in LookaheadIterator(chords):
        if not bar.lookahead:
            for i in instruments:
                drum_track[i].add_event(beat, ending_pattern[i])
        elif sections.has_key(bar.index + 2):
            for i in instruments:
                drum_track[i].add_event(beat, fill_pg[i].gen())
        else:
            for i in instruments:
                drum_track[i].add_event(beat, pg[i].gen())
        beat += 4.0

    drum_seq = Sequence()
    for i in instruments:
        drum_seq = drum_seq.merge(midify_pat_seq(drum_track[i], drum_map[i]))
    
    return perf_seq(drum_seq)
