# Copyright (c) 2007 Andrew Choi.  All rights reserved.

# This is emphatically NOT free/GPL software.

# Permission for the use of this code is granted only for research,
# educational, and non-commercial purposes.

# Redistribution of this code or its parts in any form without
# permission, with or without modification, is prohibited.
# Modifications include, but are not limited to, translation to other
# programming languages and reuse of tables, constant definitions, and
# API's defined in it.

# There is no restriction on the use of the output generated by this
# software.

# THIS SOFTWARE IS PROVIDED BY ANDREW CHOI "AS IS" AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED.  IN NO EVENT SHALL ANDREW CHOI BE LIABLE FOR ANY DIRECT,
# INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
#  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
# STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
# IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.


from mjb.utils import are_approximately_equal, fractional_part, unroll_repeats
from mjb.objects import Pattern
from toe.midi.objects import MIDINote, Sequence

from bass import gen_bass
from drums import gen_drums
from piano import gen_piano

COUNTOFF_MIDINOTE = MIDINote(75)  # Claves
COUNTOFF_VELOCITY = 100
COUNTOFF_PATTERN = Pattern('|x...x...|x.x.x.x.|')
COUNTOFF_LENGTH = 8.0

SWING = 2.3
swing_adj = SWING / (SWING + 1.0) - 0.5

def swing(seq):
    result = Sequence()

    for time, event in seq:
        mn, dur, vel = event

        if are_approximately_equal(fractional_part(time), 0.5):
            result.add_event(time + swing_adj, (mn, dur - swing_adj, vel))
        else:
            result.add_event(time, (mn, dur, vel))
            
    return result

def countoff_sequence():
    result = Sequence()
    time = 0.0
    for beat, duration in COUNTOFF_PATTERN.beats:
        f = 4.0 / COUNTOFF_PATTERN.div_per_bar
        result.add_event(time + beat * f, (COUNTOFF_MIDINOTE, duration * f, COUNTOFF_VELOCITY))
    return result

def gen(chart):
    meter = chart['style']['meter']
    if meter != 4:
        raise ValueError('Meter not equal to 4: %d' % meter)

    chart2 = unroll_repeats(chart)

    bass = swing(gen_bass(chart2)).shift(COUNTOFF_LENGTH)

    drums = countoff_sequence().merge(swing(gen_drums(chart2)).shift(COUNTOFF_LENGTH))

    piano = swing(gen_piano(chart2)).shift(COUNTOFF_LENGTH)

    return [(bass, 'Bass', 2, 33), (piano, 'Piano', 1, 1), (drums, 'Drums', 10, 1)]
