# Copyright (c) 2007 Andrew Choi.  All rights reserved.

# This is emphatically NOT free/GPL software.

# Permission for the use of this code is granted only for research,
# educational, and non-commercial purposes.

# Redistribution of this code or its parts in any form without
# permission, with or without modification, is prohibited.
# Modifications include, but are not limited to, translation to other
# programming languages and reuse of tables, constant definitions, and
# API's defined in it.

# There is no restriction on the use of the output generated by this
# software.

# THIS SOFTWARE IS PROVIDED BY ANDREW CHOI "AS IS" AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED.  IN NO EVENT SHALL ANDREW CHOI BE LIABLE FOR ANY DIRECT,
# INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
#  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
# STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
# IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.


from random import random

from toe.core.objects import Chord, Interval
from toe.midi.objects import MIDINote, Sequence

from mjb.utils import LookaheadIterator
from mjb.objects import Pattern
from mjb.generators import PG

LONG_VELOCITY = 70
SHORT_VELOCITY = 90

four_beat_no_pickup_pg = PG([('|x-.x....|', 20),
                             ('|x-.x--..|', 20),
                             ('|x-.x----|', 20),
                             ('|x---.x--|', 20),
                             ('|x-----x.|', 20),
                             ('|.x..x-..|', 10),
                             ('|.x...x..|', 20),
                             ('|..x-.x--|', 10),
                             ('|..x-....|', 10),
                             ('|...x....|', 10),
                             ('|.....x..|', 10)])

four_beat_pickup_pg = PG([('x|-..x....|', 10),
                          ('x|--...x..|', 10),
                          ('x|----..x.|', 10),
                          ('x|--------|', 10)])

two_beat_first_no_pickup_pg = PG([('|x.......|', 20),
                                  ('|x-......|', 20),
                                  ('|..x-....|', 10),
                                  ('|.x......|', 10)])

two_beat_first_pickup_pg = PG([('x|----....|', 10),
                               ('x|-.......|', 20)])

two_beat_second_no_pickup_pg = PG([('|x-......|', 30),
                                   ('|.x......|', 10),
                                   ('|.x--....|', 10)])

two_beat_second_pickup_pg = PG([('x|-.......|', 10),
                                ('x|........|', 30)])

ending_pattern = Pattern('|x---|')

TWO_BEAT_FIRST_USE_PICKUP_PROB = 0.25
TWO_BEAT_SECOND_USE_PICKUP_PROB = 0.3
FOUR_BEAT_USE_PICKUP_PROB = 0.25

MIDICHORD_VAR_1_PROB = 0.5

LOWEST_LH_MIDINOTE = MIDINote('B2')
LOWEST_RH_MIDINOTE = MIDINote('A3')

VAR_1_LH = [[1], [6, 7]]
VAR_1_RH = [[3, 4], [13, 11, 5], [9]]

VAR_2_LH = [[1], [3, 4]]
VAR_2_RH = [[6, 7], [9], [13, 11, 5]]

def at_section_start(sections, bar_index, beat):
    return sections.has_key(bar_index + 1) and beat == 0

def prev_pattern_allows_pickup(prev_pattern, prev_dur):
    if not prev_pattern:
        return False
    
    latest = 0
    for t, d in prev_pattern.beats:
        if t + d > latest:
            latest = t + d
    return latest * 4 / prev_pattern.div_per_bar < prev_dur

def prev_chord_allows_pickup(prev_chord, chord):
    if not prev_chord:
        return False

    return prev_chord.root == chord.root or prev_chord.root == chord.root + Interval('5') or prev_chord.root == chord.root + Interval('2')

def use_pickup_by_random(dur, beat):
    if dur == 2:
        if beat == 0:
            return random() < TWO_BEAT_FIRST_USE_PICKUP_PROB
        else:
            return random() < TWO_BEAT_SECOND_USE_PICKUP_PROB
    else:
        return random() < FOUR_BEAT_USE_PICKUP_PROB

def midinote_immediately_above(mn, note):
    mn2 = MIDINote(note, mn.octave)
    if mn2 > mn:
        return mn2
    else:
        return MIDINote(note, mn.octave + 1)

# A "voicing variation" is specified by a pair (left-hand and
# right-hand) of lists of lists of "interval types".  Only the first
# interval type in each sublist which is present in the chord is
# chosen.  If none of the interval types in the sublist is present,
# that sublist is skipped.
def voice_midichord(chord, lh, rh):
    result = []

    def choose_midinotes(initial_lowest, rules):
        curr_lowest = initial_lowest
        for l in rules:
            for i in l:
                if i == 1:
                    note = chord.root
                else:
                    note = chord.nth(i)
                if note:
                    curr_lowest = midinote_immediately_above(curr_lowest, note)
                    result.append(curr_lowest)
                    break
        
    choose_midinotes(LOWEST_LH_MIDINOTE, lh)
    choose_midinotes(LOWEST_RH_MIDINOTE, rh)

    return result

def midichord_var_1(chord):
    return voice_midichord(chord, VAR_1_LH, VAR_1_RH)

def midichord_var_2(chord):
    return voice_midichord(chord, VAR_2_LH, VAR_2_RH)

def choose_midichord(chord, prev):
    if not prev:
        if random() < MIDICHORD_VAR_1_PROB:
            return midichord_var_1(chord)
        else:
            return midichord_var_2(chord)
    else:
        midichord1 = midichord_var_1(chord)
        midichord2 = midichord_var_2(chord)
        if abs(midichord1[-1] - prev[-1]) < abs(midichord2[-1] - prev[-1]):
            return midichord1
        else:
            return midichord2            

def midify_pat_seq(seq):
    midi_seq = Sequence()
    prev_midichord = None

    for time, event in seq:
        pattern, chord, pattern_duration = event

        midichord = choose_midichord(chord, prev_midichord)
        
        for beat, duration in pattern.beats:
            f = 4.0 / pattern.div_per_bar
            midi_seq.add_event(time + beat * f, (midichord, duration * f))

        prev_midichord = midichord
    
    return midi_seq

def perf_seq(seq):
    result = Sequence()

    for time, event in seq:
        mn, dur = event

        if dur <= 1.0:
            result.add_event(time, (mn, dur, SHORT_VELOCITY))
        else:
            result.add_event(time, (mn, dur, LONG_VELOCITY))

    return result

def gen_piano(chart):
    sections = chart['sections']
    chords = chart['chords']

    prev_pattern = None
    prev_chord = None
    prev_dur = None

    pat_seq = Sequence()

    for bar in LookaheadIterator(chords):
        beat = 0
        for cd in LookaheadIterator(bar.curr):
            c, dur = cd.curr
            chord = Chord(c)
            if not bar.lookahead and not cd.lookahead or dur not in [2, 4]:
                pattern = Pattern('|x' + '-' * (dur - 1) + '|')
            else:
                c1 = not at_section_start(sections, bar.index, beat)
                c2 = prev_pattern_allows_pickup(prev_pattern, prev_dur)
                c3 = prev_chord_allows_pickup(prev_chord, chord)
                c4 = use_pickup_by_random(dur, beat)
                if c1 and c2 and c3 and c4:
                    if dur == 2:
                        if beat == 0:
                            pattern = two_beat_first_pickup_pg.gen()
                        else:
                            pattern = two_beat_second_pickup_pg.gen()
                    else:
                        pattern = four_beat_pickup_pg.gen()
                else:
                    if dur == 2:
                        if beat == 0:
                            pattern = two_beat_first_no_pickup_pg.gen()
                        else:
                            pattern = two_beat_second_no_pickup_pg.gen()
                    else:
                        pattern = four_beat_no_pickup_pg.gen()
            
            pat_seq.add_event(bar.index * 4 + beat, (pattern, chord, dur))
            beat += dur

            prev_pattern = pattern
            prev_chord = chord 
            prev_dur = dur
            
    return perf_seq(midify_pat_seq(pat_seq))
