# Copyright (c) 2007 Andrew Choi.  All rights reserved.

# This is emphatically NOT free/GPL software.

# Permission for the use of this code is granted only for research,
# educational, and non-commercial purposes.

# Redistribution of this code or its parts in any form without
# permission, with or without modification, is prohibited.
# Modifications include, but are not limited to, translation to other
# programming languages and reuse of tables, constant definitions, and
# API's defined in it.

# There is no restriction on the use of the output generated by this
# software.

# THIS SOFTWARE IS PROVIDED BY ANDREW CHOI "AS IS" AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED.  IN NO EVENT SHALL ANDREW CHOI BE LIABLE FOR ANY DIRECT,
# INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
#  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
# STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
# IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.


__all__ = ['biab_chord_type_to_toe', 'toe_chord_type_to_biab']

from sets import Set
from sys import maxint

from toe.core.objects import ChordType

biab_chord_types = [
    ChordType(':3.5'),  # "", 1
    ChordType(':3.5'),  # "Maj", 2
    ChordType(':3.b5'),  # "b5", 3
    ChordType(':3.#5'),  # "aug", 4
    ChordType(':3.5.6'),  # "6", 5
    ChordType(':3.5.7'),  # "Maj7", 6
    ChordType(':3.5.7.9'),  # "Maj9", 7
    ChordType(':3.5.7.9.#11'),  # "Maj9#11", 8
    ChordType(':3.5.7.9.#11.13'),  # "Maj13#11", 9
    ChordType(':3.5.7.9.11.13'),  # "Maj13", 10
    ChordType(':5.7.9'),  # "Maj9(no 3)", 11
    ChordType(':3.#5'),  # "+", 12
    ChordType(':3.#5.7'),  # "Maj7#5", 13
    ChordType(':3.5.6.9'),  # "69", 14
    ChordType(':2.5'),  # "2", 15
    ChordType(':b3.5'),  # "m", 16
    ChordType(':b3.#5'),  # "maug", 17
    ChordType(':b3.5.7'),  # "mMaj7", 18
    ChordType(':b3.5.b7'),  # "m7", 19
    ChordType(':b3.5.b7.9'),  # "m9", 20
    ChordType(':b3.5.b7.9.11'),  # "m11", 21
    ChordType(':b3.5.b7.9.11.13'),  # "m13", 22
    ChordType(':b3.5.6'),  # "m6", 23
    ChordType(':b3.#5'),  # "m#5", 24
    ChordType(':b3.#5.b7'),  # "m7#5", 25
    ChordType(':b3.5.6.9'),  # "m69", 26
    None,  # "?", 27
    None,  # "?", 28
    None,  # "?", 29
    None,  # "?", 30
    None,  # "?", 31
    ChordType(':b3.b5.b7'),  # "m7b5", 32
    ChordType(':b3.b5'),  # "dim", 33
    None,  # "?", 34
    None,  # "?", 35
    None,  # "?", 36
    None,  # "?", 37
    None,  # "?", 38
    None,  # "?", 39
    ChordType(':5'),  # "5", 40
    None,  # "?", 41
    None,  # "?", 42
    None,  # "?", 43
    None,  # "?", 44
    None,  # "?", 45
    None,  # "?", 46
    None,  # "?", 47
    None,  # "?", 48
    None,  # "?", 49
    None,  # "?", 50
    None,  # "?", 51
    None,  # "?", 52
    None,  # "?", 53
    None,  # "?", 54
    None,  # "?", 55
    ChordType(':3.#5.b7'),  # "7+", 56
    ChordType(':3.#5.b7.9'),  # "9+", 57
    ChordType(':3.#5.b7.9.11.13'),  # "13+", 58
    None,  # "?", 59
    None,  # "?", 60
    None,  # "?", 61
    None,  # "?", 62
    None,  # "?", 63
    ChordType(':3.5.b7'),  # "7", 64
    ChordType(':3.5.b7.9.11.13'),  # "13", 65
    ChordType(':3.5.b7.b13'),  # "7b13", 66
    ChordType(':3.5.b7.#11'),  # "7#11", 67
    ChordType(':3.5.b7.9.#11.13'),  # "13#11", 68
    ChordType(':3.5.b7.#11.b13'),  # "7#11b13", 69
    ChordType(':3.5.b7.9'),  # "9", 70
    None,  # "?", 71
    ChordType(':3.5.b7.9.b13'),  # "9b13", 72
    ChordType(':3.5.b7.9.#11'),  # "9#11", 73
    ChordType(':3.5.b7.9.#11.13'),  # "13#11", 74
    ChordType(':3.5.b7.9.#11.b13'),  # "9#11b13", 75
    ChordType(':3.5.b7.b9'),  # "7b9", 76
    ChordType(':3.5.b7.b9.11.13'),  # "13b9", 77
    ChordType(':3.5.b7.b9.b13'),  # "7b9b13", 78
    ChordType(':3.5.b7.b9.#11'),  # "7b9#11", 79
    ChordType(':3.5.b7.b9.#11.13'),  # "13b9#11", 80
    ChordType(':3.5.b7.b9.#11.b13'),  # "7b9#11b13", 81
    ChordType(':3.5.b7.#9'),  # "7#9", 82
    ChordType(':3.5.b7.#9.11.13'),  # "13#9", 83
    ChordType(':3.5.b7.#9.b13'),  # "7#9b13", 84
    ChordType(':3.5.b7.9.#11'),  # "9#11", 85
    ChordType(':3.5.b7.#9.#11.13'),  # "13#9#11", 86
    ChordType(':3.5.b7.#9.#11.b13'),  # "7#9#11b13", 87
    ChordType(':3.b5.b7'),  # "7b5", 88
    ChordType(':3.b5.b7.9.11.13'),  # "13b5", 89
    ChordType(':3.b5.b7.b13'),  # "7b5b13", 90
    ChordType(':3.b5.b7.9'),  # "9b5", 91
    ChordType(':3.b5.b7.9.b13'),  # "9b5b13", 92
    ChordType(':3.b5.b7.b9'),  # "7b5b9", 93
    ChordType(':3.b5.b7.b9.11.13'),  # "13b5b9", 94
    ChordType(':3.b5.b7.b9.b13'),  # "7b5b9b13", 95
    ChordType(':3.b5.b7.#9'),  # "7b5#9", 96
    ChordType(':3.b5.b7.#9.11.13'),  # "13b5#9", 97
    ChordType(':3.b5.b7.#9.b13'),  # "7b5#9b13", 98
    ChordType(':3.#5.b7'),  # "7#5", 99
    ChordType(':3.#5.b7.9.11.13'),  # "13#5", 100
    ChordType(':3.#5.b7.#11'),  # "7#5#11", 101
    ChordType(':3.#5.b7.9.#11.13'),  # "13#5#11", 102
    ChordType(':3.#5.b7.9'),  # "9#5", 103
    ChordType(':3.#5.b7.9.#11'),  # "9#5#11", 104
    ChordType(':3.#5.b7.b9'),  # "7#5b9", 105
    ChordType(':3.#5.b7.b9.11.13'),  # "13#5b9", 106
    ChordType(':3.#5.b7.b9.#11'),  # "7#5b9#11", 107
    ChordType(':3.#5.b7.b9.#11.13'),  # "13#5b9#11", 108
    ChordType(':3.#5.b7.#9'),  # "7#5#9", 109
    ChordType(':3.#5.b7.#9.#11.13'),  # "13#5#9#11", 110
    ChordType(':3.#5.b7.#9.#11'),  # "7#5#9#11", 111
    ChordType(':3.#5.b7.#9.#11.13'),  # "13#5#9#11", 112
    ChordType(':3.b5.#5.b7.b9.#9'),  # "7alt", 113
    None,  # "?", 114
    None,  # "?", 115
    None,  # "?", 116
    None,  # "?", 117
    None,  # "?", 118
    None,  # "?", 119
    None,  # "?", 120
    None,  # "?", 121
    None,  # "?", 122
    None,  # "?", 123
    None,  # "?", 124
    None,  # "?", 125
    None,  # "?", 126
    None,  # "?", 127
    ChordType(':4.5.b7'),  # "7sus", 128
    ChordType(':4.5.b7.9.11.13'),  # "13sus", 129
    ChordType(':4.5.b7.b13'),  # "7susb13", 130
    ChordType(':4.5.b7.#11'),  # "7sus#11", 131
    ChordType(':4.5.b7.9.#11.13'),  # "13sus#11", 132
    ChordType(':4.5.b7.#11.b13'),  # "7sus#11b13", 133
    ChordType(':4.5.b7.9'),  # "9sus", 134
    None,  # "?", 135
    ChordType(':4.5.b7.9.b13'),  # "9susb13", 136
    ChordType(':4.5.b7.9.#11'),  # "9sus#11", 137
    ChordType(':4.5.b7.9.#11.13'),  # "13sus#11", 138
    ChordType(':4.5.b7.9.#11.b13'),  # "9sus#11b13", 139
    ChordType(':4.5.b7.b9'),  # "7susb9", 140
    ChordType(':4.5.b7.b9.11.13'),  # "13susb9", 141
    ChordType(':4.5.b7.b9.13'),  # "7susb913", 142
    ChordType(':4.5.b7.b9.#11'),  # "7susb9#11", 143
    ChordType(':4.5.b7.b9.#11.13'),  # "13susb9#11", 144
    ChordType(':4.5.b7.b9.#11.b13'),  # "7susb9#11b13", 145
    ChordType(':4.5.b7.#9'),  # "7sus#9", 146
    ChordType(':4.5.b7.#9.11.13'),  # "13sus#9", 147
    ChordType(':4.5.b7.#9.b13'),  # "7sus#9b13", 148
    ChordType(':4.5.b7.9.#11'),  # "9sus#11", 149
    ChordType(':4.5.b7.#9.#11.13'),  # "13sus#9#11", 150
    ChordType(':4.5.b7.#9.#11.b13'),  # "7sus#9#11b13", 151
    ChordType(':4.b5.b7'),  # "7susb5", 152
    ChordType(':4.b5.b7.9.11.13'),  # "13susb5", 153
    ChordType(':4.b5.b7.b13'),  # "7susb5b13", 154
    ChordType(':4.b5.b7.9'),  # "9susb5", 155
    ChordType(':4.b5.b7.9.b13'),  # "9susb5b13", 156
    ChordType(':4.b5.b7.b9'),  # "7susb5b9", 157
    ChordType(':4.b5.b7.b9.11.13'),  # "13susb5b9", 158
    ChordType(':4.b5.b7.b9.b13'),  # "7susb5b9b13", 159
    ChordType(':4.b5.b7.#9'),  # "7susb5#9", 160
    ChordType(':4.b5.b7.#9.11.13'),  # "13susb5#9", 161
    ChordType(':4.b5.b7.#9.b13'),  # "7susb5#9b13", 162
    ChordType(':4.#5.b7'),  # "7sus#5", 163
    ChordType(':4.#5.b7.9.11.13'),  # "13sus#5", 164
    ChordType(':4.#5.b7.#11'),  # "7sus#5#11", 165
    ChordType(':4.#5.b7.9.#11.13'),  # "13sus#5#11", 166
    ChordType(':4.#5.b7.9'),  # "9sus#5", 167
    ChordType(':4.#5.b7.9.#11'),  # "9sus#5#11", 168
    ChordType(':4.#5.b7.b9'),  # "7sus#5b9", 169
    ChordType(':4.#5.b7.b9.11.13'),  # "13sus#5b9", 170
    ChordType(':4.#5.b7.b9.#11'),  # "7sus#5b9#11", 171
    ChordType(':4.#5.b7.b9.#11.13'),  # "13sus#5b9#11", 172
    ChordType(':4.#5.b7.#9'),  # "7sus#5#9", 173
    ChordType(':4.#5.b7.#9.#11.13'),  # "13sus#5#9#11", 174
    ChordType(':4.#5.b7.#9.#11'),  # "7sus#5#9#11", 175
    ChordType(':4.#5.b7.#9.#11.13'),  # "13sus#5#9#11", 176
    ChordType(':4'),  # "4", 177
    None,  # "?", 178
    None,  # "?", 179
    None,  # "?", 180
    None,  # "?", 181
    None,  # "?", 182
    None,  # "?", 183
    ChordType(':4.5'),  # "sus", 184
    # no chord names above 185
    ]

biab_chord_type_reverse_map = {}

i = 1
for chord_type in biab_chord_types:
    if chord_type:
        biab_chord_type_reverse_map[chord_type._interval_list_string()] = i
    i += 1

def biab_chord_type_to_toe(type_code):
    i = type_code - 1
    if i < 0 or i >= len(biab_chord_types):
        raise ValueError('BiaB chord type code out of range: %d' % type_code)

    chord_type = biab_chord_types[i]
    
    if not chord_type:
        raise ValueError('Unknown BiaB chord type code: %d' % type_code)
    
    return str(chord_type)

def toe_chord_type_to_biab(chord_type_name):
    chord_type = ChordType(chord_type_name)
    i = biab_chord_type_reverse_map.get(chord_type._interval_list_string())
    if i != None:
        return i, None
    else:
        diff = maxint
        i = 1
        for chord_type2 in biab_chord_types:
            if chord_type2:
                new_diff = chord_type_difference(chord_type, chord_type2)
                if new_diff < diff:
                    best = i
                    diff = new_diff
            i += 1
            
        return best, (str(chord_type), str(biab_chord_types[best - 1]))

def chord_type_difference(ct1, ct2):
    il1 = Set(ct1._interval_list_string().split('.'))
    il2 = Set(ct2._interval_list_string().split('.'))
    return len(il1.symmetric_difference(il2))
