(*

The main purpose of the release of this code is to demonstrate the
author's T2 algorithm for harmonic analysis of jazz chord sequences.

Permission for the use of this code is granted only for research,
educational, and non-commercial purposes.

Redistribution of this code or its parts in source, binary,
and any other form without permission, with or without modification,
is prohibited.  Modifications include, but are not limited to,
translation to other programming languages and reuse of tables,
constant definitions, and API's defined in it.

Andrew Choi is not liable for any losses or damages caused by the use
of this software.

Copyright 2008 Andrew Choi.
http://www.sixthhappiness.ca/T2/index.html

*)

open Num

open Structana
open Segment

(* Types of dominant resolutions.  *)

type 'a res =
    Nores
  | Domres of 'a
  | Subres of 'a
  | Passthrures of 'a * 'a

let anno_non_dom k c =
  let r = Roman.of_intvl (Note.sub_note (Simplchord.root c) k) in
    let rc = Romanchord.of_root_kind r (Simplchord.kind c) in
      Romanchord.to_string rc

let anno_dia k c =
  let r = Roman.of_intvl (Note.sub_note (Simplchord.root c) k) in
    let rc = Romanchord.of_root_kind r (Simplchord.kind c) in
      if List.mem rc dia || List.mem rc m_root || List.mem rc m_dia then
	Romanchord.to_string rc
      else
	failwith "Chord isn't diatonic"

let dom_annotation k c nt nc =
  let i = Note.sub_note (Simplchord.root c) (Simplchord.root nc) in
    if Intvl.equiv i i5 then
      let r = Roman.of_intvl (Note.sub_note (Simplchord.root nc) k) in
	let sr = Roman.to_string r in
	  if sr = "I" then
	    "V7", Domres (nt)
	  else
	    "V7/" ^ sr, Domres (nt)
    else if Intvl.equiv i ib2 then
      let r = Roman.of_intvl (Note.sub_note (Simplchord.root nc) k) in
	let sr = Roman.to_string r in
	  if sr = "I" then
	    "subV7", Subres (nt)
	  else
	    "subV7/" ^ sr, Subres (nt)
    else
      failwith "Doesn't resolve"

let rec anno_dom k c optnae =
  if Simplchord.kind c <> "7" then
    failwith "Not dom chord"
  else
    match optnae with
      None ->  (* No next chord: treat like other chords.  *)
	anno_non_dom k c, Nores
    | Some nae ->
	match nae with
	  Chord (nt, _, nc) ->
	    dom_annotation k c nt nc
	| Turnaround (ael, _) ->
	    anno_dom k c (Some (List.hd ael))
	| Twofive (two, five) ->
	    begin
	      try
		anno_dom k c (Some two)
	      with _ ->
		anno_dom k c (Some five)
	    end	  
	| Passthru (p, m) ->
	    begin
	      match p, m with
		Chord (pt, _, _), Chord (mt, _, mc) ->
		  let anno, res = dom_annotation k c mt mc in
		    anno, Passthrures (pt, mt)
	      | _ ->
		  failwith "Main of Passthru can only be Chord variant"
	    end
	| Extdom ael ->
	    anno_dom k c (Some (List.hd ael))

let anno_dr_dom k c =
  if Simplchord.kind c <> "7" then
    failwith "Not dom chord"
  else
    let r = Roman.of_intvl (Note.sub_note (Simplchord.root c) k) in
      (Roman.to_string r) ^ "7"

let anno_chord k t c nae =
  if Simplchord.kind c <> "7" then
    anno_non_dom k c, Nores
  else
    try
      anno_dom k c nae
    with _ ->
      try
	"(" ^ anno_dr_dom k c ^ ")", Nores
      with _ ->
	failwith "No annotation for chord"

let tl_anno_chord k ae nae =
  match ae with
    Chord (t, d, c) ->
      begin
	try
	  let anno, res = anno_chord k t c nae in
	    Chord (t, d, Simplchord.orig c, anno, res)
	with _ ->
	  Chord (t, d, Simplchord.orig c, "?", Nores)
      end
  | _ ->
      failwith "tl_anno_chord handles only Chord variant"

let tl_anno_turnaround k aelist nae =
  let rec tl_anno_turnaround1 = function
      a :: b :: tl ->
	tl_anno_chord k a (Some b) :: tl_anno_turnaround1 (b :: tl)
    | a :: [] ->
	tl_anno_chord k a nae :: []
    | _ ->
	failwith "Extdom contains less than one ae"
  in
    Turnaround (tl_anno_turnaround1 aelist, [])

let tl_anno_passthru k (p, m) nae =
  match p, m with
    Chord _, Chord _ ->
      Passthru (tl_anno_chord k p (Some m), tl_anno_chord k m nae)
  | _, _ ->
      failwith "Passthru contains non-Chord variant"

let tl_anno_two k two =
  match two with
    Chord (t, d, c) ->
      begin
	try
	  Chord (t, d, Simplchord.orig c, anno_dia k c, Nores)
	with _ ->
	  Chord (t, d, Simplchord.orig c, "", Nores)
      end
  | _ ->
      failwith "tl_anno_two handles only Chord variant"

let tl_anno_twofive k (two, five) nae =
  match two, five with
    Chord _, Chord _ ->
      Twofive (tl_anno_two k two, tl_anno_chord k five nae)
  | Chord _, Passthru (p, m) ->
      Twofive (tl_anno_two k two, tl_anno_passthru k (p, m) nae)
  | _, _ ->
      failwith "Twofive contains non-Chord and non-Passthru variant"

let null_anno_chord k ae nae =
  match ae with
    Chord (t, d, c) ->
      begin
	try
	  let anno, res = anno_chord k t c nae in
	    Chord (t, d, Simplchord.orig c, "", res)
	with _ ->
	  Chord (t, d, Simplchord.orig c, "?", Nores)
      end
  | _ ->
      failwith "null_anno_chord handles only Chord variant"

let null_anno_passthru k (p, m) nae =
  match p, m with
    Chord _, Chord _ ->
      Passthru (null_anno_chord k p (Some m), null_anno_chord k m nae)
  | _, _ ->
      failwith "Passthru contains non-Chord variant"

let null_anno_twofive k (two, five) nae =
  match two, five with
    Chord _, Chord _ ->
      Twofive (null_anno_chord k two None, null_anno_chord k five nae)
  | Chord _, Passthru (p, m) ->
      Twofive (null_anno_chord k two None, null_anno_passthru k (p, m) nae)
  | _, _ ->
      failwith "Twofive contains non-Chord and non-Passthru variant"

let rec null_anno k ae nae =
  match ae with
    Chord _ ->
      null_anno_chord k ae nae
  | Twofive (two, five) ->
      null_anno_twofive k (two, five) nae
  | Passthru (pt, main) -> 
      null_anno_passthru k (pt, main) nae
  | _ ->
      failwith "Extdom may contain only chords, twofives, and passthrus"

let anno_extdom_first k ae nae =
  match ae with
    Chord (t, d, c) ->
      begin
	try
	  let anno, res = anno_dom k c nae in
	    Chord (t, d, Simplchord.orig c, "(" ^ anno ^ ")", res)
	with _ ->
	  failwith "No annotation for chord"
      end
  | Twofive (two, five) ->
      tl_anno_twofive k (two, five) nae
  | _ ->
      failwith "Head of Extdom must be Chord or Twofive"

let anno_extdom_last k ae nae =
  match ae with
    Chord _ ->
      tl_anno_chord k ae nae
  | Twofive (two, five) ->
      begin
	match two, five with
	  Chord _, Chord _ ->
	    Twofive (null_anno k two None, tl_anno_chord k five nae)
	| Chord _, Passthru (p, m) ->
	    Twofive (null_anno k two None, tl_anno_passthru k (p, m) nae)
	| _, _ ->
	    failwith "Twofive contains non-Chord and non-Passthru variant"
      end
  | Passthru (p, m) ->
      begin
	match p, m with
	  Chord _, Chord _ ->
	    Passthru (null_anno k p (Some m), tl_anno_chord k m nae)
	| _, _ ->
	    failwith "Passthru contains non-Chord variant"
      end
  | _ ->
      failwith "Head of Extdom must be Chord or Twofive"

let rec anno_extdom_tl k nae = function
    a :: [] ->
      anno_extdom_last k a nae :: []
  | a :: b :: tl ->
      null_anno k a (Some b) :: anno_extdom_tl k nae (b :: tl)
  | _ ->
      failwith "Extdom contains less than two ae"

let tl_anno_extdom k aelist nae =
  let tl_anno_extdom1 = function
      a :: b :: tl ->
	anno_extdom_first k a (Some b) :: anno_extdom_tl k nae (b :: tl)
    | _ ->
	failwith "Extdom contains less than one ae"
  in
    Extdom (tl_anno_extdom1 aelist)

(* Annotate analysis element ae at the top-level in key k where the
next analysis element is nae and whether the previous analysis element
has resolved to this one is given by the boolean value r. *)

let tl_anno k ae nae =
  match ae with
    Chord _ -> tl_anno_chord k ae nae
  | Turnaround (aelist, _) -> tl_anno_turnaround k aelist nae
  | Twofive (two, five) -> tl_anno_twofive k (two, five) nae
  | Passthru (pt, main) -> tl_anno_passthru k (pt, main) nae
  | Extdom aelist -> tl_anno_extdom k aelist nae

let rec anno k = function
    a :: b :: tl ->
      tl_anno k a (Some b) :: anno k (b :: tl)
  | a :: [] ->
      tl_anno k a None :: []
  | [] ->
      []

let annotate p =
  List.map
    (fun (k, aea) ->
      let keyroot k =
	let l = String.length k in
	  if k.[l - 1] = 'm' then String.sub k 0 (l - 1) else k
      in
	k, anno (Note.of_string (keyroot k)) (Array.to_list aea))
    p

(*let ana fn =
  let chart = Toereader.read fn in
    let a = Array.of_list (structana (List.concat chart.Chart.chords)) in
      let cm = gen_cm a and mcm = m_gen_cm a and n = Array.length a in
	let p = postprocess (buildpart (optpart (gen_d cm mcm a) n) a n) in
	  fn, annotate p*)
