(*

The main purpose of the release of this code is to demonstrate the
author's T2 algorithm for harmonic analysis of jazz chord sequences.

Permission for the use of this code is granted only for research,
educational, and non-commercial purposes.

Redistribution of this code or its parts in source, binary,
and any other form without permission, with or without modification,
is prohibited.  Modifications include, but are not limited to,
translation to other programming languages and reuse of tables,
constant definitions, and API's defined in it.

Andrew Choi is not liable for any losses or damages caused by the use
of this software.

Copyright 2008 Andrew Choi.
http://www.sixthhappiness.ca/T2/index.html

*)

open Num
open Big_int

open Structana
open Segment
open Annotate

type twofivetype =
    Domtwofive of num * num
  | Subtwofive of num * num

type brackettype =
    Solidbracket of int
  | Dottedbracket of int

type arrowtype =
    Solidarrow of int
  | Dottedarrow of int
  | Passthru1arrow of int
  | Passthru2arrow of int

let linewidth = 80

let rec start_time ae =
  match ae with
    Chord (t, _, _, _, _) -> t
  | Turnaround (ael, _) -> start_time (List.hd ael)
  | Twofive (t, f) -> start_time t
  | Passthru (p, m) -> start_time p
  | Extdom ael -> start_time (List.hd ael)

let start_time_ael ael =
  start_time (List.hd ael)

let key_centers p =
  List.map (fun (k, part) -> k, (start_time_ael part)) p

let is_dom_twofive tc fc =
  Intvl.equiv (Note.sub_note (Chord.root tc) (Chord.root fc)) i5

let rec twofive ae =
  match ae with
  | Twofive (t, f) ->
      begin
	match t, f with
	  Chord (tt, _, tc, _, _), Chord (ft, _, fc, _, _) ->
	    if is_dom_twofive tc fc then
	      [Domtwofive (tt, ft)]
	    else
	      [Subtwofive (tt, ft)]
	| Chord (tt, _, tc, _, _), Passthru (_, Chord (ft, _, fc, _, _)) ->
	    if is_dom_twofive tc fc then
	      [Domtwofive (tt, ft)]
	    else
	      [Subtwofive (tt, ft)]
	| _, _ ->
	    failwith "Twofive contains non-Chord and non-Passthru variant"
      end
  | Extdom ael ->
      List.concat (List.map twofive ael)
  | _ ->
      []

let twofive_ael ael =
  List.concat (List.map twofive ael)

let twofives p =
  List.concat (List.map (fun (_, part) -> twofive_ael part) p)

let rec serialize ae =
  match ae with
    Chord (t, d, c, a, r) -> [(t, d, Chord.to_string c, a, r)]
  | Turnaround (ael, _) -> List.concat (List.map serialize ael)
  | Twofive (t, f) -> (serialize t) @ (serialize f)
  | Passthru (p, m) -> (serialize p) @ (serialize m)
  | Extdom ael -> List.concat (List.map serialize ael)

let serialize_ael ael =
  List.concat (List.map serialize ael)

let serialize_part p =
  List.concat (List.map (fun (_, part) -> serialize_ael part) p)

let bar_partition sap =
  let rec loop dur res = function
      (t, d, c, a, r) :: tl ->
	let d2 = d +/ dur in
	  if d2 </ (Int 4) then
	    loop d2 ((t, d, c, a, r) :: res) tl
	  else if d2 =/ (Int 4) then
	    loop (Int 0) ((t, d, c, a, r) :: res) tl
	  else
	    let dr = d2 -/ (Int 4) in
	      let dl = d -/ dr in
		let ael = t, dl, c, a, r
		and aer = t +/ dl, dr, "/", "", Nores
		in
		  loop (Int 0) (ael :: res) (aer :: tl)
    | [] ->
	List.rev res
  in
    loop (Int 0) [] sap

let gcd n1 n2 =
  num_of_big_int (gcd_big_int (big_int_of_num n1) (big_int_of_num n2))

let gcd_of_durations b =
  List.fold_left gcd (Int 4) (List.map (fun (_, d, _, _, _) -> d) b)

let bar_equalize b =
  let pulse = gcd_of_durations b in
    let rec loop res = function
	(t, d, c, a, r) :: tl ->
	  if d >/ pulse then
	    let d2 = d -/ pulse in
	      let ael = t, pulse, c, a, r
	      and aer = t +/ pulse, d2, "/", "", Nores
	      in
		loop (ael :: res) (aer :: tl)
	  else
	    loop ((t, d, c, a, r) :: res) tl
      | [] ->
	  List.rev res
    in
      loop [] (List.rev b)

let beat_equalize sap =
  let rec loop dur bar res = function
	(t, d, c, a, r) :: tl ->
	let d2 = d +/ dur in
	  if d2 =/ (Int 4) then
	    let b = bar_equalize ((t, d, c, a, r) :: bar) in
	      loop (Int 0) [] (res @ b) tl
	  else
	    loop d2 ((t, d, c, a, r) :: bar) res tl
    | [] ->
	res
  in
    loop (Int 0) [] [] sap

module OrderedNum =
struct
  type t = num
  let compare x y = compare_num x y
end

module NumMap = Map.Make(OrderedNum) 

let timestamp_map esapa =
  let map = ref NumMap.empty in
    Array.iteri (fun i (t, _, _, _, _) -> map := NumMap.add t i !map) esapa;
    !map

let gen_chords esapa =
  Array.map (fun (_, _, c, a, _) -> c, a) esapa

let gen_barlines esapa =
  Array.map (fun (t, _, _, _, _) -> mod_num t (Int 4) =/ (Int 0)) esapa

let gen_arrows esapa tsm =
  let n = Array.length esapa in
    let arrows = Array.make n None in
      for i = n - 1 downto 0 do
	match esapa.(i) with
	  (_, _, _, _, r) ->
	    begin
	      match r with
		Nores ->
		  ()
	      | Domres t ->
		  arrows.(i) <- Some (Solidarrow (NumMap.find t tsm))
	      | Subres t ->
		  arrows.(i) <- Some (Dottedarrow (NumMap.find t tsm))
	      | Passthrures (t1, t2) ->
		  let i1 = NumMap.find t1 tsm and i2 = NumMap.find t2 tsm in
		    arrows.(i) <- Some (Passthru1arrow i1);
		    arrows.(i1) <- Some (Passthru2arrow i2)
	    end
      done;
      arrows

let gen_keys kc tsm n =
  let keys = Array.make n None in
    List.iter (fun (k, t) -> keys.(NumMap.find t tsm) <- Some k) kc;
    keys

let gen_brackets tf tsm n =
  let brackets = Array.make n None in
    List.iter
      (fun twofive ->
	match twofive with
	  Domtwofive (t1, t2) ->
	    let i1 = NumMap.find t1 tsm and i2 = NumMap.find t2 tsm in
	      brackets.(i1) <- Some (Solidbracket (i2))	      
	| Subtwofive (t1, t2) ->
	    let i1 = NumMap.find t1 tsm and i2 = NumMap.find t2 tsm in
	      brackets.(i1) <- Some (Dottedbracket (i2)))
      tf;
    brackets

let gen_dots chords keys =
  let n = Array.length chords in
    let dots = Array.make n 0 in
      for i = 0 to n - 1 do
	let c, a = chords.(i) in
	  dots.(i) <- max (String.length c) (String.length a);
	  match keys.(i) with
	    None -> ()
	  | Some k -> dots.(i) <- max dots.(i) (String.length k)
      done;
      dots

let gen_spaces barlines arrows dots =
  let n = Array.length barlines in
    let spaces = Array.make n 0 in
      for i = 0 to n - 1 do
	spaces.(i) <- 1;
	if i < n - 1 && barlines.(i + 1) then
	  spaces.(i) <- 3;
	match arrows.(i) with
	  None -> ()
	| Some a ->
	    match a with
	      Solidarrow j ->
		if j = i + 1 then
		  spaces.(i) <- 3
	    | Dottedarrow j ->
		if j = i + 1 then
		  spaces.(i) <- 3
	    | Passthru1arrow j ->
		if j = i + 1 then
		  spaces.(i) <- max 2 spaces.(i)
	    | Passthru2arrow j ->
		if j = i + 1 then
		  spaces.(i) <- max (4 - dots.(i)) spaces.(i)
      done;
      spaces.(n - 1) <- 3;
      spaces

let gen_bars dots spaces barlines =
  let n = Array.length dots in
    let rec loop w (i, j) res =
      if j >= n - 1 then
	List.rev (((i, j), (dots.(j) + spaces.(j) + w)) :: res)
      else if barlines.(j + 1) then
	loop 0 (j + 1, j + 1) (((i, j), (dots.(j) + spaces.(j) + w)) :: res)
      else
	loop (dots.(j) + spaces.(j) + w) (i, j + 1) res
    in
      loop 0 (0, 0) []

let gen_lines bars linewidth =
  let rec loop (a, b) w res = function
      ((i, j), bw) :: tl ->
	if w + bw > linewidth then
	  loop (i, j) 0 ((a, b) :: res) (((i, j), bw) :: tl)
	else
	  loop (a, j) (w + bw) res tl
    | [] ->
	List.rev ((a, b) :: res)
  in
    loop (0, 0) 0 [] bars

let dvi_output ap =
  let esap = beat_equalize (bar_partition (serialize_part ap))
  and kc = key_centers ap
  and tf = twofives ap
  in
    let esapa = Array.of_list esap in
      let tsm = timestamp_map esapa and n = Array.length esapa in
	let chords = gen_chords esapa
	and barlines = gen_barlines esapa
	and arrows = gen_arrows esapa tsm
	and keys = gen_keys kc tsm n
	and brackets = gen_brackets tf tsm n
	in
	  chords, barlines, arrows, keys, brackets

let output_arrows arrows active_arrow s e dots spaces =
  let ar = ref active_arrow in
    begin
      match !ar with
	None -> Printf.printf "  "
      | Some b ->
	  begin
	    match b with
	      Solidarrow j ->
		if j = s then
		  begin
		    Printf.printf "->";
		    ar := None
		  end
		else
		  Printf.printf "--"
	    | Dottedarrow j ->
		if j = s then
		    begin
		      Printf.printf ".>";
		      ar := None
		    end
		else
		  Printf.printf ".."
	    | Passthru1arrow j ->
		Printf.printf "--";
		if j = s then
		  ar := None
	    | Passthru2arrow j ->
		Printf.printf "..";
		if j = s then
		  ar := None
	  end
    end;
    for i = s to e do
      let w = dots.(i) and u = if i = e then spaces.(i) - 1 else spaces.(i) in
	begin
	  match !ar with
	    None ->
	      begin
		ar := arrows.(i);
		begin
		  match !ar with
		    None ->
		      Printf.printf "%-*s" w ""
		  | Some b ->
		      begin
			match b with
			  Passthru2arrow _ ->
			    Printf.printf "%s" (":" ^ (String.make (w - 1) '.'))
			| _ ->
			    Printf.printf "%-*s" w ""
		      end
		end
	      end
	  | Some b ->
	      begin
		match b with
		  Solidarrow _ ->
		    Printf.printf "%s" (String.make w '-')
		| Dottedarrow _ ->
		    Printf.printf "%s" (String.make w '.')
		| Passthru1arrow _ ->
		    Printf.printf "%s" (String.make w '-')
		| Passthru2arrow _ ->
		    Printf.printf "%s" (String.make w '.')
	      end
	end;
	match !ar with
	  None ->
	    Printf.printf "%-*s" u ""
	| Some b ->
	    begin
	      match b with
		Solidarrow j ->
		  if i <> e && j = i + 1 then
		    begin
		      Printf.printf "%s" ((String.make (u - 1) '-') ^ ">");
		      ar := None
		    end
		  else
		    Printf.printf "%s" (String.make u '-')
	      | Dottedarrow j ->
		  if i <> e && j = i + 1 then
		    begin
		      Printf.printf "%s" ((String.make (u - 1) '.') ^ ">");
		      ar := None
		    end
		  else
		    Printf.printf "%s" (String.make u '.')
	      | Passthru1arrow j ->
		  Printf.printf "%s" (String.make u '-');
		  if i <> e && j = i + 1 then
		    ar := None
	      | Passthru2arrow j ->
		  if i <> e && j = i + 1 then
		    begin
		      Printf.printf "%s" ((String.make (u - 1) '.') ^ ">");
		      ar := None
		    end
		  else
		    Printf.printf "%s" (String.make u '.')
	    end
    done;
    Printf.printf "\n";
    !ar

let output_keys keys s e dots spaces =
  Printf.printf " ";
  for i = s to e do
    match keys.(i) with
      None ->
	Printf.printf "%-*s%-*s" dots.(i) "" spaces.(i) ""
    | Some k ->
	Printf.printf "%-*s%-*s" dots.(i) (k ^ ":") spaces.(i) ""
  done;
  Printf.printf "\n"
  
let output_annotations chords s e dots spaces =
  Printf.printf "  ";
  for i = s to e do
    Printf.printf "%-*s%-*s" dots.(i) (snd chords.(i)) spaces.(i) ""
  done;
  Printf.printf "\n"
  
let output_chords chords s e dots spaces barlines =
  let n = Array.length chords in
    Printf.printf "| ";
    for i = s to e do
      Printf.printf "%-*s" dots.(i) (fst chords.(i));
      if i >= n - 1 || barlines.(i + 1) then
	Printf.printf "%-*s" spaces.(i) " |"
      else
	Printf.printf "%-*s" spaces.(i) ""
    done;
    Printf.printf "\n"

let output_brackets brackets active_bracket s e dots spaces =
  let ab = ref active_bracket in
    begin
      match !ab with
	None -> Printf.printf "  "
      | Some b ->
	  begin
	    match b with
	      Solidbracket j -> Printf.printf "__"
	    | Dottedbracket j -> Printf.printf ".."
	  end
    end;
    for i = s to e do
      let w = dots.(i) and u = if i = e then spaces.(i) - 1 else spaces.(i) in
	begin
	  match !ab with
	    None ->
	      begin
		match brackets.(i) with
		  None ->
		    Printf.printf "%-*s" w ""
		| Some b ->
		    begin
		      match b with
			Solidbracket j ->
			  Printf.printf "%s" ("\\" ^ (String.make (w - 1) '_'))
		      | Dottedbracket j ->
			  Printf.printf "%s" ("\\" ^ (String.make (w - 1) '.'))
		    end;
		    ab := brackets.(i)
	      end
	  | Some b ->
	      begin
		match b with
		  Solidbracket j ->
		    if j = i then
		      begin
			Printf.printf "%-*s" w "/";
			ab := None
		      end
		    else
		      Printf.printf "%s" (String.make w '_')
		| Dottedbracket j ->
		    if j = i then
		      begin
			Printf.printf "%-*s" w "/";
			ab := None
		      end
		    else
		      Printf.printf "%s" (String.make w '.')
	      end
	end;
	match !ab with
	  None ->
	    Printf.printf "%-*s" u ""
	| Some b ->
	    begin
	      match b with
		Solidbracket j ->
		  Printf.printf "%s" (String.make u '_')
	      | Dottedbracket j ->
		  Printf.printf "%s" (String.make u '.')
	    end
    done;
    Printf.printf "\n";
    !ab

let text_output chords barlines arrows keys brackets =
  let dots = gen_dots chords keys in
    let spaces = gen_spaces barlines arrows dots in
      let lines = gen_lines (gen_bars dots spaces barlines) (linewidth - 2) in
	let ab = ref None and ar = ref None in
	  List.iter
	    (fun (s, e) ->
	      output_keys keys s e dots spaces;
	      ar := output_arrows arrows !ar s e dots spaces;
	      output_annotations chords s e dots spaces;
	      output_chords chords s e dots spaces barlines;
	      ab := output_brackets brackets !ab s e dots spaces;
	      Printf.printf "\n")
	  lines

(*
let testdir = "/Users/choi/Projects/testfiles/coker-toe"

let ana fn =
  Sys.chdir testdir;
  let chart = Toereader.read fn in
    let a = Array.of_list (structana (List.concat chart.Chart.chords)) in
      let clsm = gen_clsm a and m_clsm = m_gen_clsm a in
	let dur = gen_dur a in
	  let dm = gen_dm clsm and m_dm = m_gen_dm m_clsm in
	    let n = Array.length a in
	      let em = gen_em clsm m_clsm dur n
	      and m_em = m_gen_em clsm m_clsm dur n in
		let d2m = gen_d2m dm em keys 
		and m_d2m = gen_d2m m_dm m_em m_keys in
		  let p = buildpart (optpart (gen_c d2m m_d2m a) n) a n in
		    let ap = annotate p in
		      let chords, bl, arrows, keys, brackets = dvi_output ap in
			text_output chords bl arrows keys brackets
*)
