(*

The main purpose of the release of this code is to demonstrate the
author's T2 algorithm for harmonic analysis of jazz chord sequences.

Permission for the use of this code is granted only for research,
educational, and non-commercial purposes.

Redistribution of this code or its parts in source, binary,
and any other form without permission, with or without modification,
is prohibited.  Modifications include, but are not limited to,
translation to other programming languages and reuse of tables,
constant definitions, and API's defined in it.

Andrew Choi is not liable for any losses or damages caused by the use
of this software.

Copyright 2008 Andrew Choi.
http://www.sixthhappiness.ca/T2/index.html

*)

(* Roman numerals (or Roman) seem to look like /intervals/ at first
glance, but they're really more like /notes/, because they serve as
roots of chords, and in analysis, we add intervals to them to get
other roman numerals, and subtract pairs of them to get intervals.  *)

(* It might have been more appropriate to define Roman as a class that
inherits from Note.  But since the rest of TOE uses no object-oriented
langauge features, it is defined here as a type that encapsulates a
Note.  I might just as well, because Roman /works like/ a Note but is
not /a kind of/ Note.  I can always change it if I later find out I
don't like it.  *)

open Utils

type t = { n: Note.t }

let roman_re =
  let alt = "\\(\\(##\\|#\\|bb\\|b\\)?\\)"
  and num = "\\(III\\|II\\|IV\\|VII\\|VI\\|V\\|I\\)" in
    Str.regexp ("^" ^ alt ^ num ^ "$")

let num_names = [|"I"; "II"; "III"; "IV"; "V"; "VI"; "VII"|]
let alt_names = [|"bb"; "b"; ""; "#"; "##"|]

let of_string s =
  if Str.string_match roman_re s 0 then
    let a = Str.matched_group 1 s and n = Str.matched_group 3 s in
      let n = index_array num_names n
      and alt = (index_array alt_names a) - 2 in
	{ n = Note.of_n_alt n alt }
  else
    failwith ("Invalid roman: " ^ s)

let to_string roman =
  let n, alt = Note.to_n_alt roman.n in
    if alt < -2 || alt > 2 then
      failwith ("Invalid alt: " ^ (string_of_int alt))
    else
      alt_names.(alt + 2) ^ num_names.(n)

let pp formatter x = Format.pp_print_string formatter (to_string x)

let of_intvl intvl =
  let i, alt = Intvl.to_i_alt intvl in
    if alt < -2 || alt > 2 then
      failwith ("Invalid alt: " ^ (string_of_int alt))
    else if i < 0 || alt > 6 then
      failwith ("Invalid i: " ^ (string_of_int i))
    else
      { n = Note.of_n_alt i alt }

let to_intvl roman =
  let n, alt = Note.to_n_alt roman.n in
    Intvl.of_i_alt n alt

let compare r1 r2 =
  Note.compare r1.n r2.n

let sub_roman r1 r2 =
  Note.sub_note r1.n r2.n

let equiv r1 r2 =
  Note.semi r1.n = Note.semi r2.n
