(*

The main purpose of the release of this code is to demonstrate the
author's T2 algorithm for harmonic analysis of jazz chord sequences.

Permission for the use of this code is granted only for research,
educational, and non-commercial purposes.

Redistribution of this code or its parts in source, binary,
and any other form without permission, with or without modification,
is prohibited.  Modifications include, but are not limited to,
translation to other programming languages and reuse of tables,
constant definitions, and API's defined in it.

Andrew Choi is not liable for any losses or damages caused by the use
of this software.

Copyright 2008 Andrew Choi.
http://www.sixthhappiness.ca/T2/index.html

*)

open Num

open Structana


(* Change this if your testfiles are in another directory.  *)

let testdir = "/Users/choi/Projects/testfiles/coker-toe"

let romankey_re =
  let intvl = "\\(\\(##\\|#\\|bb\\|b\\)?\\(III\\|II\\|IV\\|VII\\|VI\\|V\\|I\\)\\)" in
    Str.regexp ("^" ^ intvl ^ "\\(m?\\)" ^ "$")

type ae_class = Root | Dia | Pridom | Subpridom | Secdom | Subsecdom | Dim | Blues | Modint | Ta | Unknown

let keys = ["Gb"; "Db"; "Ab"; "Eb"; "Bb"; "F"; "C"; "G"; "D"; "A"; "E"; "B"]
let m_keys = ["Eb"; "Bb"; "F"; "C"; "G"; "D"; "A"; "E"; "B"; "F#"; "C#"; "G#"]

(* Major key harmony. *)

let dia = List.map Romanchord.of_string ["IMaj"; "IIm"; "IIm6"; "IIm7"; "IIIm"; "IIIm7"; "IVMaj"; "VIm"; "VIm6"; "VIm7"; "VImMaj7"; "VIIm7b5"]

let pridom = List.map Romanchord.of_string ["V7"]

let subpridom = List.map Romanchord.of_string ["bII7"; "#I7"]

let secdom = List.map Romanchord.of_string ["VI7"; "VII7"; "I7"; "II7"; "III7"]

(* The last two should really be categorized separately as "weak subdominants".  See Nettles and Graf.  *)

let subsecdom = List.map Romanchord.of_string ["bIII7"; "bV7"; "bVI7"; "IV7"; "bVII7"]

let blues = List.map Romanchord.of_string ["IV7"]

let modint = List.map Romanchord.of_string ["Im7"; "IIm7b5"; "bIIIMaj"; "IVm"; "IVm6"; "IVm7"; "IVmMaj7"; "Vm"; "Vm6"; "Vm7"; "bVIMaj"; "bVII7"]

(* Minor key harmony. *)

let m_root = List.map Romanchord.of_string ["Im"; "Im6"; "Im7"; "ImMaj7"]

let m_dia = List.map Romanchord.of_string ["IIm"; "IIm7b5"; "IIm7"; "bIIIMaj"; "IVm"; "IVm7"; "Vm"; "Vm7"; "bVIMaj"; "VIm7b5"; "bVIIdim"; "bVIIm7b5"]

let m_pridom = List.map Romanchord.of_string ["V7"]

let m_subpridom = List.map Romanchord.of_string ["bII7"; "#I7"]

let m_secdom = List.map Romanchord.of_string ["VI7"; "bVII7"; "I7"; "II7"; "bIII7"; "IV7"]

let m_subsecdom = List.map Romanchord.of_string ["bII7"; "II7"; "bV7"; "bVI7"; "VII7"]


(* Utilities: triangular 2-D arrays and string maps.  *)

let make_tri n initval =
  Array.init n (fun i -> Array.make (n - i) initval)

module OrderedString =
struct
  type t = string
  let compare x y = String.compare x y
end

module StringMap = Map.Make(OrderedString) 


(* AE Classification. *)

let ae_romanize ae k =
  let c = ae_c (lastchord ae) in
    let r = Roman.of_intvl (Note.sub_note (Simplchord.root c) k) in
      Romanchord.of_root_kind r (Simplchord.kind c)

let ae_classify k is_minor rules ae =
  match ae with
    Turnaround (_, keys) ->
      if List.mem ((Note.to_string k) ^ is_minor) keys then
	Ta
      else
	Unknown
  | _ ->
      try
	let rc = ae_romanize ae k in
	  let rec classify1 = function
	      (test, res) :: tl ->
		if test rc ae then
		  res
		else
		  classify1 tl
	    | _ ->
		Unknown
	  in
	    classify1 rules
      with _ ->
	Unknown

let mem2 l e _ =
  List.mem e l

let bluestest rc ae =
  match ae with
    Chord _ ->
      (* Don't classify ii-V's etc. as "Blues".  *)
      List.mem rc blues
  | _ ->
      false

(* Classification rules are a list of pairs of tests and classes.
Each test is run in turn and the first to succeed causes the
corresponding class to be returned.  *)

type ruletype = (Romanchord.t -> (Num.num * Num.num * Simplchord.t) Structana.ae -> bool) * ae_class

let rules =
  ([((fun rc _ -> rc = Romanchord.of_string "IMaj"), Root);
    ((mem2 dia), Dia);
    ((mem2 pridom), Pridom);
    ((mem2 subpridom), Subpridom);
    ((mem2 secdom), Secdom);
    ((mem2 subsecdom), Subsecdom);
    ((fun rc _ -> Romanchord.kind rc = "dim"), Dim);
    (bluestest, Blues);
    ((mem2 modint), Modint)] : ruletype list)

let m_rules =
  ([((mem2 m_root), Root);
    ((mem2 m_dia), Dia);
    ((mem2 m_pridom), Pridom);
    ((mem2 m_subpridom), Subpridom);
    ((mem2 m_secdom), Secdom);
    ((mem2 m_subsecdom), Subsecdom);
    ((fun rc _ -> Romanchord.kind rc = "dim"), Dim)] : ruletype list)

let aea_classify_by_key rules aea k is_minor =
  Array.map (ae_classify k is_minor rules) aea

let gen_clsm a =
  List.fold_left
    (fun m k ->
      let a2 = aea_classify_by_key rules a (Note.of_string k) "" in
	StringMap.add k a2 m)
    StringMap.empty
    keys

let m_gen_clsm a =
  List.fold_left
    (fun m k ->
      let a2 = aea_classify_by_key m_rules a (Note.of_string k) "m" in
	StringMap.add k a2 m)
    StringMap.empty
    m_keys


(* AE Duration. *)

let rec ae_dur = function
    Chord (_, d, _) -> d
  | Turnaround (ael, _) ->
      (List.fold_left (fun sum ae -> sum +/ (ae_dur ae)) (Int 0) ael)
  | Twofive (t, f) -> ae_dur t +/ ae_dur f
  | Passthru (p, m) -> ae_dur p +/ ae_dur m
  | Extdom ael -> List.fold_left (fun sum ae -> sum +/ (ae_dur ae)) (Int 0) ael

let gen_dur a =
  let n = Array.length a in
    let dur = Array.make n (Int 0) in
      for i = 0 to n - 1 do
	dur.(i) <- ae_dur a.(i)
      done;
      dur


(* Cost Function.  *)

let cost a =
  match a with
    Root -> -6.
  | Dia -> -5.
  | Pridom -> -4.
  | Subpridom -> -3.
  | Secdom -> -2.
  | Subsecdom -> -1.
  | Dim -> 0.
  | Blues -> 0.
  | Modint -> 0.
  | Ta -> 0.
  | Unknown -> infinity

let modcost = 1000.
let singletoncost = 1000000.

(* k.(i).(j) is just the sum of the costs associated with ae_i, ...,
ae_j, respectively.  Tabulated to keep within O(n^2) time.  *)

let gen_k a =
  let n = Array.length a in
    let k = make_tri n 0. in
      for i = 0 to n - 1 do
	k.(i).(0) <- cost a.(i);
      done;
      for i = 0 to n - 2 do
	for j = i + 1 to n - 1 do
	  k.(i).(j - i) <- k.(i).(j - 1 - i) +. k.(j).(0)
	done
      done;
      k

(* r.(i).(j) is true iff the subsegment ae_i, ..., ae_j contains one
or more Roots or Turnaround.  *)

let gen_r a =
  let n = Array.length a in
    let r = make_tri n false in
      for i = 0 to n - 1 do
	r.(i).(0) <- a.(i) = Root || a.(i) = Ta
      done;
      for i = 0 to n - 2 do
	for j = i + 1 to n - 1 do
	  r.(i).(j - i) <- r.(i).(j - i - 1) || r.(j).(0)
	done
      done;
      r

(* f.(i).(j) is true iff the subsegment ae_i, ..., ae_j does not end
in a dominant chord.  *)

let dominants = [Pridom; Subpridom; Secdom; Subsecdom]

let gen_f a =
  let n = Array.length a in
    let f = make_tri n false in
      for i = 0 to n - 1 do
	for j = i to n - 1 do
	  f.(i).(j - i) <- not (List.mem a.(j) dominants) 
	done
      done;
      f

(* u.(i).(j) is true iff the subsegment ae_i, ..., ae_j does not
contain any Unknown.  *)

let gen_u a =
  let n = Array.length a in
    let u = make_tri n false in
      for i = 0 to n - 1 do
	u.(i).(0) <- a.(i) <> Unknown
      done;
      for i = 0 to n - 2 do
	for j = i + 1 to n - 1 do
	  u.(i).(j - i) <- u.(i).(j - i - 1) && u.(j).(0)
	done
      done;
      u

(* s combines r and u and f.  *)

let gen_s a =
  let n = Array.length a in
    let s = make_tri n false in
      let r = gen_r a and u = gen_u a and f = gen_f a in
	for i = 0 to n - 1 do
	  for j = i to n - 1 do
	    s.(i).(j - i) <- r.(i).(j - i) && u.(i).(j - i) && f.(i).(j - i)
	  done
	done;
	s

(* d.(i).(j) is the cost of ae_i, ..., ae_j in the "current" key
center, given by the sum of costs according to the classes of its AEs
if the subsegment contains at least a root, does not contain unknowns,
and does not end in a dominant chord.  Its value is infinity
otherwise.  *)

let gen_d a =
  let n = Array.length a in
    let k = gen_k a and s = gen_s a in
      let d = make_tri n (-1.) in
	for i = 0 to n - 1 do
	  for j = i to n - 1 do
	    d.(i).(j - i) <- if s.(i).(j - i) then k.(i).(j - i) else infinity
	  done
	done;
	d

(* Generate key-to-d-matrix maps.  *)

let gen_dm clsm =
  List.fold_left
    (fun m k -> StringMap.add k (gen_d (StringMap.find k clsm)) m)
    StringMap.empty
    keys

let m_gen_dm m_clsm =
  List.fold_left
    (fun m k -> StringMap.add k (gen_d (StringMap.find k m_clsm)) m)
    StringMap.empty
    m_keys


(* Additional processing on "d maps" to detect embedded segments in
related keys.  Z represents the list of spans containing only AEs in
z_classes that contain a Root and not ending in dominants and passing
z_test.  Intervals within and outside the spans are labelled with odd
and even ID's, respectively.  Each interval is assigned a different ID
that increases with index values.  *)

let gen_z a aok dur z_classes z_test =
  let n = Array.length a in
    let z = Array.make (n + 1) 0 in
      let zi = ref 0 and b = ref (-1) and r = ref false in
	let fill r b i =
	  let k = ref (i - 1) in
	    while !k >= !b && List.mem a.(!k) dominants do
	      k := !k - 1
	    done;
	    if !r && z_test a dur !b !k then
	      begin
		for j = !b to !k do
		  z.(j) <- !zi + 1
		done;
		for j = !k + 1 to i do
		  z.(j) <- !zi + 2
		done;
		zi := !zi + 2;
	      end
	    else
	      begin
		for j = !b to i do
		  z.(j) <- !zi
		done
	      end;
	    b := -1;
	    r := false
	in
	  for i = 0 to n - 1 do
	    if List.mem a.(i) z_classes && aok.(i) <> Root then
	      begin
		if !b = -1 then
		  b := i;
		if a.(i) = Root then
		  r := true
	      end
	    else
	      begin
		if !b <> -1 then
		  fill r b i
		else
		  z.(i) <- !zi
	      end
	  done;
	  if !b <> -1 then
	    fill r b n;
	  Array.sub z 0 n

(* Related Keys.  *)

let z_classes = [Root; Pridom]

let totdur dur i j =
  let tot = ref (Int 0) in
    for k = i to j do
      tot := !tot +/ dur.(k)
    done;
    !tot

let z_test _ (* a *) dur i j =
  totdur dur i j >=/ (Int 8) 

let root_others_dur a dur i j =
  let r = ref (Int 0) and o = ref (Int 0) in
    for k = i to j do
      if a.(k) = Root then
	r := !r +/ dur.(k)
      else
	o := !o +/ dur.(k)
    done;
    (!r, !o)

let z4l_classes = [Root; Pridom; Dia; Secdom; Subpridom; Subsecdom]

let z4s_classes = [Root; Pridom]

let z4l_test a dur i j =
  let (r, o) = root_others_dur a dur i j in
    (Int 4) */ r >=/ o && r +/ o >=/ (Int 32)

let z4s_test a dur i j =
  let (r, o) = root_others_dur a dur i j in
    (Int 2) */ r >=/ o && o >/ (Int 0) && r +/ o >=/ (Int 8)

(* Generate e matrix for segments in a given key that are not allowed
to overlap subsegments in a related key (AEs classified in a).  This
is the method for disqualifying subsegments of AEs for the current key
center when *long* subspans that can be analyzed in related keys have
been detected.  The presence of such long subspans indicates that the
analysis of original key should not appear within these subspans nor
overlap them.  *)

let gen_e_nol a aok dur z_classes z_test =
  let n = Array.length a in
    let e_rk = make_tri n false in
      let z = gen_z a aok dur z_classes z_test in
	for i = 0 to n - 1 do
	  for j = i to n - 1 do
	    e_rk.(i).(j - i) <- z.(i) = z.(j)
	  done
	done;
	e_rk

(* Generate e matrix for segments in a given key for which subsegments
in a related key (AEs classified in a) are not allowed to appear at
its ends.  This is the method for *short* subspans analyzeable in
related keys.  Since these subspans are short, the span in the
original key may in fact subsume them.  But when they appear at the
end of a span in the original key (followed or preceded by a
modulation to or from another key center), we tend to want to identify
them and separate them from the original key.  This is what is
typically done in the Coker appendix anyway.  *)

let gen_e_ne a aok dur z_classes z_test =
  let n = Array.length a in
    let e_ne = make_tri n false in
      let z = gen_z a aok dur z_classes z_test in
	for i = 0 to n - 1 do
	  for j = i to n - 1 do
	    if z.(i) = z.(j) then
	      e_ne.(i).(j - i) <- true
	    else if z.(i) mod 2 = 0 && z.(j) mod 2 = 0 then
	      e_ne.(i).(j - i) <- true
	    else
	      e_ne.(i).(j - i) <- false	      
	  done
	done;
	e_ne

let tri_and t1 t2 n =
  let res = make_tri n false in
    for i = 0 to n - 1 do
      for j = i to n - 1 do
	res.(i).(j - i) <- t1.(i).(j - i) && t2.(i).(j - i)
      done
    done;
    res

let romankey_to_intvl_and_is_minor s =
  if Str.string_match romankey_re s 0 then
    let roman = Str.matched_group 1 s and is_minor = Str.matched_group 4 s in
      let intvl = Roman.to_intvl (Roman.of_string roman) in
	(intvl, is_minor = "m")
  else
    failwith ("Invalid romankey: " ^ s)

let relatedkeyrules =
  List.map
    (fun (r, f, c, t) -> (romankey_to_intvl_and_is_minor r), f, c, t)
    [("bIII", gen_e_nol, z_classes, z_test);
     ("bVI", gen_e_nol, z_classes, z_test);
     ("IV", gen_e_nol, z4l_classes, z4l_test);
     ("IV", gen_e_ne, z4s_classes, z4s_test);
     ("VIm", gen_e_nol, z4l_classes, z4l_test);
     ("VIm", gen_e_ne, z4s_classes, z4s_test);
     ("Im", gen_e_ne, z_classes, z_test);
     ("IIm", gen_e_ne, z_classes, z_test);
     ("IIIm", gen_e_ne, z_classes, z_test);
     ("IVm", gen_e_ne, z_classes, z_test);
     ("Vm", gen_e_ne, z_classes, z_test)]

let m_relatedkeyrules =
  List.map
    (fun (r, f, c, t) -> (romankey_to_intvl_and_is_minor r), f, c, t)
    [("bIII", gen_e_nol, z_classes, z_test);
     ("bVI", gen_e_nol, z_classes, z_test);
     ("IIm", gen_e_ne, z_classes, z_test);
     ("Vm", gen_e_ne, z_classes, z_test);
     ("IVm", gen_e_nol, z4l_classes, z4l_test);
     ("IVm", gen_e_ne, z4s_classes, z4s_test)]

let key_add_intvl_keyname k i keys =
  let k2 = Note.add_intvl (Note.of_string k) i in
    List.find (fun k3 -> Note.semi (Note.of_string k3) = Note.semi k2) keys

let key_intvl_is_minor_to_cls k (intvl, is_minor) clsm m_clsm =
  if is_minor then
    StringMap.find (key_add_intvl_keyname k intvl m_keys) m_clsm
  else
    StringMap.find (key_add_intvl_keyname k intvl keys) clsm

(* e.(i).(j) is true iff the segment ae_i, ..., ae_j is not
disqualified from being used in a segmentation solution because it
contains a subsegment that can be analyzed in a related key.  Process
each rule in relatedkeyrules and generate an "e" matrix for that
rules.  Then accumulate and return the logical "AND" of all these
"e"matrices.  *)

let gen_e clsm m_clsm dur k n =
  let e = ref (make_tri n true) and clsok = StringMap.find k clsm in
    List.iter
      (fun ((invtl, is_minor), gen_e_func, z_classes, z_test) ->
	let cls = key_intvl_is_minor_to_cls k (invtl, is_minor) clsm m_clsm in
	  let e2 = gen_e_func cls clsok dur z_classes z_test in
	    e := tri_and !e e2 n)
      relatedkeyrules;
    !e

let m_gen_e clsm m_clsm dur k n =
  let e = ref (make_tri n true) and clsok = StringMap.find k m_clsm in
    List.iter
      (fun ((invtl, is_minor), gen_e_func, z_classes, z_test) ->
	let cls = key_intvl_is_minor_to_cls k (invtl, is_minor) clsm m_clsm in
	  let e2 = gen_e_func cls clsok dur z_classes z_test in
	    e := tri_and !e e2 n)
      m_relatedkeyrules;
    !e

let gen_em clsm m_clsm dur n =
  List.fold_left
    (fun m k -> StringMap.add k (gen_e clsm m_clsm dur k n) m)
    StringMap.empty
    keys

let m_gen_em clsm m_clsm dur n =
  List.fold_left
    (fun m k -> StringMap.add k (m_gen_e clsm m_clsm dur k n) m)
    StringMap.empty
    m_keys

let gen_d2m dm em keys =
  List.fold_left
    (fun m k ->
      StringMap.add
	k
	(let d = StringMap.find k dm and e = StringMap.find k em in
	  let n = Array.length d in
	    let d2 = make_tri n 0. in
	      for i = 0 to n - 1 do
		for j = i to n - 1 do
		  d2.(i).(j - i) <-
		    if e.(i).(j - i) then d.(i).(j - i) else infinity
		done
	      done;
	      d2)
	m)
    StringMap.empty
    keys

(* Last resort when a single ae must form its own segment.  Most often
caused by unforeseen turnarounds and chord patterns.  *)

let singletonsimplchordkinds = ["m6"; "m7"; "mMaj7"; "m"; "m7b5"]

let singletonkey ae =
  match ae with
    Turnaround (_, keys) ->
      List.hd keys
  | _ ->
      let sc = ae_c (lastchord ae) in
	let r = Note.to_string (Simplchord.root sc) in
	  if List.mem (Simplchord.kind sc) singletonsimplchordkinds then
	    r ^ "m"
	  else
	    r


(* Let n be the length of the ae array and d_ij, 0 <= i <= j <= n - 1,
be the minimal cost of the subsequence at positions i, ..., j in the
ae array among all keys.  *)

let gen_c dm m_dm a =
  let n = Array.length a in
    let mincostandkey i j =
      let mincost = ref infinity
      and minkey =  ref "" in
	List.iter
	  (fun currkey ->
	    let currcost = (StringMap.find currkey dm).(i).(j - i) in
	      if currcost < !mincost then
		begin
		  mincost := currcost;
		  minkey := currkey
		end)
	  keys;
	List.iter
	  (fun currkey ->
	    let currcost = (StringMap.find currkey m_dm).(i).(j - i) in
	      if currcost < !mincost then
		begin
		  mincost := currcost;
		  minkey := currkey ^ "m"
		end)
	  m_keys;
	!mincost, !minkey
    in
      let c = make_tri n (0., "") in
	for i = 0 to n - 1 do
	  for j = i to n - 1 do
	    c.(i).(j - i) <- mincostandkey i j
	  done
	done;
	for i = 0 to n - 1 do
	  let ci0, _ = c.(i).(0) in
	    if ci0 = infinity then
	      c.(i).(0) <- singletoncost, singletonkey a.(i)
	done;
	c

let optpart c n =
  let mc = Array.make n 0.
  and ik = Array.make n (0, "") in
    for i = 0 to n - 1 do
      let c0, k0 = c.(0).(i) in
	let mincost = ref c0
	and minkey = ref k0
	and minindex = ref 0 in
	  for k = 1 to i do
	    let ck, kk = c.(k).(i - k) in
	      let currcost = ck +. mc.(k - 1) +. modcost in
		if currcost < !mincost then
		  begin
		    mincost := currcost;
		    minkey := kk;
		    minindex := k
		  end
	  done;
	  mc.(i) <- !mincost;
	  ik.(i) <- !minindex, !minkey
    done;
    ik

let buildpart op a n =
  let rec loop i res =
    if i < 0 then
      res
    else
      let i2, k = op.(i) in
	loop (i2 - 1) ((k, Array.sub a i2 (i - i2 + 1)) :: res)
  in
    loop (n - 1) []

let ana fn =
  Sys.chdir testdir;
  let chart = Toereader.read fn in
    let a = Array.of_list (structana (List.concat chart.Chart.chords)) in
      let clsm = gen_clsm a and m_clsm = m_gen_clsm a in
	let dur = gen_dur a in
	  let dm = gen_dm clsm and m_dm = m_gen_dm m_clsm in
	    let n = Array.length a in
	      let em = gen_em clsm m_clsm dur n
	      and m_em = m_gen_em clsm m_clsm dur n in
		let d2m = gen_d2m dm em keys 
		and m_d2m = gen_d2m m_dm m_em m_keys in
		  let p = buildpart (optpart (gen_c d2m m_d2m a) n) a n in
		    fn, p

let anaall () =
  [ana "1";
   ana "2";
   ana "3";
   ana "4";
   ana "5";
   ana "6";
   ana "7";
   ana "8";
   ana "9";
   ana "10";
   ana "11";
   ana "12";
   ana "13";
   ana "14";
   ana "15";
   ana "16";
   ana "17";
   ana "18";
   ana "19";
   ana "20";
   ana "21";
   ana "22";
   ana "23";
   ana "24";
   ana "25";
   ana "26";
   ana "27";
   ana "28";
   ana "29";
   ana "30";
   ana "31a";
   ana "31b";
   ana "32";
   ana "33";
   ana "35";
   ana "36";
   ana "37";
   ana "38";
   ana "39";
   ana "41";
   ana "42";
   ana "43";
   ana "44";
   ana "46";
   ana "47";
   ana "48";
   ana "49";
   ana "50";
   ana "51";
   ana "52";
   ana "53";
   ana "54";
   ana "55";
   ana "56";
   ana "57";
   ana "58";
   ana "61";
   ana "62";
   ana "63";
   ana "64";
   ana "65";
   ana "66";
   ana "68";
   ana "69";
   ana "70";
   ana "71";
   ana "72";
   ana "73";
   ana "74";
   ana "75";
   ana "76";
   ana "77";
   ana "78";
   ana "79";
   ana "80";
   ana "81";
   ana "82";
   ana "83"]
