(*

The main purpose of the release of this code is to demonstrate the
author's T2 algorithm for harmonic analysis of jazz chord sequences.

Permission for the use of this code is granted only for research,
educational, and non-commercial purposes.

Redistribution of this code or its parts in source, binary,
and any other form without permission, with or without modification,
is prohibited.  Modifications include, but are not limited to,
translation to other programming languages and reuse of tables,
constant definitions, and API's defined in it.

Andrew Choi is not liable for any losses or damages caused by the use
of this software.

Copyright 2008 Andrew Choi.
http://www.sixthhappiness.ca/T2/index.html

*)

open Num

(* An "analysis" is represented by a list of (individual) chords,
two-fives, extended dominants, and pass-throughs.  An element of this
list has type "ae", which stands for "analysis element".  *)

type 'a ae =
    Chord of 'a
  | Turnaround of 'a ae list * string list
  | Twofive of 'a ae * 'a ae
  | Extdom of 'a ae list
  | Passthru of 'a ae * 'a ae

let romankey_re =
  let intvl = "\\(\\(##\\|#\\|bb\\|b\\)?\\(III\\|II\\|IV\\|VII\\|VI\\|V\\|I\\)\\)" in
    Str.regexp ("^" ^ intvl ^ "\\(m?\\)" ^ "$")

let ib2 = Intvl.of_string "b2"
let i5 = Intvl.of_string "5"

(* Utilities for type ae.  *)

let ae_t = function
    Chord (t, _, _) -> t
  | _ -> failwith "Can't get timestamp for complex ae"

let ae_d = function
    Chord (_, d, _) -> d
  | _ -> failwith "Can't get duration for complex ae"

let ae_c = function
    Chord (_, _, c) -> c
  | _ -> failwith "Can't get chord for complex ae"

let intvlbetw e1 e2 =
  Note.sub_note (Simplchord.root (ae_c e1)) (Simplchord.root (ae_c e2))

let rec list_last l =
  match l with
    [] -> failwith "No last element on empty list"
  | a :: [] -> a
  | _ :: tl -> list_last tl

let rec lastchord = function
    Chord _ as c -> c
  | Turnaround (ael, _) -> lastchord (list_last ael)
  | Twofive (_, five) -> lastchord five
  | Passthru (_, main) -> lastchord main
  | Extdom ael -> lastchord (list_last ael)

let rec firstchord = function
    Chord _ as c -> c
  | Turnaround (ael, _) -> firstchord (List.hd ael)
  | Twofive (two, _) -> lastchord two
  | Passthru (_, main) -> lastchord main  (* ?? *)
  | Extdom ael -> firstchord (List.hd ael)

(* The various stages of the analysis.  *)

let makeseq chords =
  let seq = ref [] in
    let f ts (c, d) =
      let dur = (Int d) in
	seq := (ts, dur, (Simplchord.of_chord_name c)) :: !seq;
	ts +/ dur
    in
      ignore (List.fold_left f (Int 0) chords);
      List.rev !seq

let mergesame chords =
  let rec loop res (t, d, c) l =
    match l with
      [] ->
	List.rev (Chord (t, d, c) :: res)
    | (t2, d2, c2) :: tl ->
	if Chord.compare (Simplchord.orig c) (Simplchord.orig c2) = 0 then
	  loop res (t, d +/ d2, c) tl
	else
	  loop (Chord (t, d, c) :: res) (t2, d2, c2) tl
  in
    match chords with
      hd :: tl -> loop [] hd tl
    | [] -> []

(* Each turnaround is represented as a list of pairs of roman chords
and their relative durations in the turnaround.  *)

let turnarounds = 
  List.map
    (fun (ta, keys) ->
      (List.map (fun (rcn, dur) -> Romanchord.of_string rcn, (Int dur)) ta), keys)
    [([("IMaj", 1); ("VIm7", 1); ("IIm7", 1); ("V7", 1)], ["I"]);
     ([("IMaj", 1); ("bIIIMaj", 1); ("bVIMaj", 1); ("bIIMaj", 1)], ["I"]);
     ([("IMaj", 1); ("bIIm7", 1); ("IIm7", 1); ("V7", 1)], ["I"]);
     ([("IMaj", 1); ("bIIdim", 1); ("IIm7", 1); ("V7", 1)], ["I"]);
     ([("IMaj", 1); ("#Idim", 1); ("IIm7", 1); ("#IIdim", 1)], ["I"]);
     ([("IIIm7", 1); ("bIIIm7", 1); ("IIm7", 1); ("bII7", 1)], ["I"; "Im"]);
     ([("IIIm7", 1); ("bIIIm7", 1); ("IIm7", 1); ("V7", 1)], ["I"; "Im"]);
     ([("IMaj", 1); ("bIIIm7", 1); ("IIm7", 1); ("bII7", 1)], ["I"]);
     ([("IMaj", 1); ("bIIIm7", 1); ("IIm7", 1); ("V7", 1)], ["I"]);
     ([("#IVm7b5", 1); ("IVm7", 1); ("IIIm7", 1); ("bIIIm7", 1)], ["I"]);
     ([("IIm7", 1); ("V7", 1); ("IIIm7", 1); ("bIIIm7", 1)], ["I"]);
     ([("bVm7b5", 1); ("IVMaj", 1); ("IIIm7", 1); ("VI7", 1)], ["I"]);
     ([("IMaj", 1); ("bIIIm7", 1); ("IIIm7", 1); ("VI7", 1)], ["I"])]

let split_at n l =
  let rec loop n left right =
    if n <= 0 then
      List.rev left, right
    else
      match right with
	a :: tl ->
	  loop (n - 1) (a :: left) tl
      | _ ->
	  failwith "Not enough elements on the list for split_at"
  in
    loop n [] l

let romankey_to_key key s =
  if Str.string_match romankey_re s 0 then
    let intvl = Str.matched_group 1 s and is_minor = Str.matched_group 4 s in
      let k = Note.add_intvl key (Roman.to_intvl (Roman.of_string intvl)) in
	(Note.to_string k) ^ is_minor
  else
    failwith ("Invalid romankey: " ^ s)

let romanize c kc =
  let r = Roman.of_intvl (Note.sub_note (Simplchord.root c) kc) in
    Romanchord.of_root_kind r (Simplchord.kind c)

let matchturnaround l ta keys =
  let ae = List.hd l and rc1, dur1 = List.hd ta in
  let r = Simplchord.root (ae_c ae)
  and i = Roman.to_intvl (Romanchord.root rc1) in
    let kc = Note.sub_intvl r i and m = (ae_d ae) // dur1 in
      if
	List.for_all2
	  (fun ae (rc, dur) ->
	    romanize (ae_c ae) kc = rc && dur */ m =/ (ae_d ae))
	  l
	  ta
      then
	Some (List.map (romankey_to_key kc) keys)
      else
	None

let parseturnaround l =
  let rec loop = function
      (ta, keys) :: ta_tl ->
	begin
	  let n = List.length ta in
	    try
	      let left, right = split_at n l in
		match matchturnaround left ta keys with
		  Some keys ->
		    Some (left, keys), right
		| None ->
		    loop ta_tl
	    with _ ->
	      loop ta_tl
	end
    | _ ->
	None, l
  in
    loop turnarounds

let rec groupturnaround l =
  match l with
    a :: tl ->
      begin
	let optta, tltl = parseturnaround l in
	  match optta with
	    None ->
	      a :: groupturnaround tl
	  | Some (ael, keys) ->
	      Turnaround (ael, keys) :: groupturnaround tltl
      end
  | [] ->
      []

(*let turnaroundkey t =
  match t with
    Turnaround ael ->
      let rec loop = function
	  ta :: ta_tl ->
	    begin
	      try
		if matchturnaround ael ta then
		  let rc1, _ = List.hd ta in
		    let r = Simplchord.root (ae_c (List.hd ael))
		    and i = Roman.to_intvl (Romanchord.root rc1) in
		      Note.sub_intvl r i
		else
		  loop ta_tl
	      with _ ->
		loop ta_tl
	    end
	| _ ->
	    failwith "No match for turnaround"
      in
	loop turnarounds
  | _ -> failwith "Can't get key of non-turnarounds"*)

let arepassthru e1 e2 =
  match e1, e2 with
    Chord _, Chord _ ->
      Intvl.equiv (intvlbetw e1 e2) ib2 && Simplchord.kind (ae_c e1) = "7"
  | _ ->
      false

let parsewithpassthru test l =
  match l with
    [] ->
      None, l
  | a :: b :: tl ->
      if test a then
	Some a, b :: tl
      else if arepassthru a b && test b then
	Some (Passthru (a, b)), tl
      else
	None, l
  | a :: [] ->
      if test a then
	Some a, []
      else
	None, l

let stress =
  let stbl = [| 15; 7; 11; 3; 13; 5; 9; 1; 14; 6; 10; 2; 12; 4; 8; 0 |] in
    fun t -> stbl.((int_of_num t) mod 16)

let aretwofive e1 e2 =
  match e1, e2 with
    Turnaround _, _ | _, Turnaround _ ->
      false
  | _, _ ->
      List.exists (Intvl.equiv (intvlbetw e1 e2)) [ib2; i5]
	&& List.mem (Simplchord.kind (ae_c e1)) ["m7"; "m7b5"]
	&& Simplchord.kind (ae_c e2) = "7"
	&& stress (ae_t e1) > stress (ae_t e2)

let rec grouptwofive l =
  match l with
    a :: tl ->
      begin
	let optfive, tltl = parsewithpassthru (aretwofive a) tl in
	  match optfive with
	    None ->
	      a :: grouptwofive tl
	  | Some five ->
	      Twofive (a, five) :: grouptwofive tltl
      end
  | [] ->
      []

let aredompair e1 e2 =
  match e1, e2 with
    Turnaround _, _ | _, Turnaround _ ->
      false
  | _, _ ->
      let l1 = lastchord e1 and l2 = lastchord e2 in
	List.exists (Intvl.equiv (intvlbetw l1 l2)) [ib2; i5]
	  && Simplchord.kind (ae_c l1) = "7"
	  && Simplchord.kind (ae_c l2) = "7"

let groupextdom l =
  let rec loop res curr l =
    match l with
      [] ->
	begin
	  match curr with
	    Extdom tfl ->
	      List.rev (Extdom (List.rev tfl) :: res)
	  | _ ->
	      List.rev (curr :: res)
	end
    | hd :: tl ->
	match curr with
	  Extdom el ->
	    begin
	      let optdom, tl2 = parsewithpassthru (aredompair (List.hd el)) l in
		match optdom with
		  None ->
		    loop ((Extdom (List.rev el)) :: res) hd tl
		| Some dom ->
		    loop res (Extdom (dom :: el)) tl2
	    end
	| _ as e ->
	    let optdom, tl2 = parsewithpassthru (aredompair e) l in
	      match optdom with
		None ->
		  loop (e :: res) hd tl
	      | Some dom ->
		  loop res (Extdom [dom; e]) tl2
  in
    match l with
      hd :: tl -> loop [] hd tl
    | [] -> []

let domresolves e1 e2 =
  match e1, e2 with
    Turnaround _, _ | _, Turnaround _ ->
      false
  | _, _ ->
      let l1 = lastchord e1 and l2 = firstchord e2 in
	Intvl.equiv (intvlbetw l1 l2) i5
	  && stress (ae_t l1) < stress (ae_t l2)
	  && Simplchord.kind (ae_c l1) = "7"

let rec grouppassthru l =
  match l with
    a :: tl ->
      begin
	let optb, tltl = parsewithpassthru (domresolves a) tl in
	  match optb with
	    None ->
	      a :: grouppassthru tl
	  | Some b ->
	      a :: b :: grouppassthru tltl
      end
  | [] ->
      []

let structana chords =
  let seq = mergesame (makeseq chords) in
    grouppassthru (groupextdom (grouptwofive (groupturnaround seq)))

(*
let test fn =
  Sys.chdir "/Users/choi/Projects/testfiles/coker-toe";
  let chart = Toereader.read fn in
    structana (List.concat chart.Chart.chords)
*)
