(*

The main purpose of the release of this code is to demonstrate the
author's T2 algorithm for harmonic analysis of jazz chord sequences.

Permission for the use of this code is granted only for research,
educational, and non-commercial purposes.

Redistribution of this code or its parts in source, binary,
and any other form without permission, with or without modification,
is prohibited.  Modifications include, but are not limited to,
translation to other programming languages and reuse of tables,
constant definitions, and API's defined in it.

Andrew Choi is not liable for any losses or damages caused by the use
of this software.

Copyright 2008 Andrew Choi.
http://www.sixthhappiness.ca/T2/index.html

*)

open Chart
open Utils

let root_names = [| "C"; "Db"; "D"; "Eb"; "E"; "F"; "Gb"; "G"; "Ab"; "A"; "Bb"; "B"; "C#"; "D#"; "F#"; "G#"; "A#" |]

let section_names = [| ' '; 'A'; 'B' |]

let read_byte s =
  int_of_char (Stream.next s)

let read_char =
  Stream.next

let read_string s n =
  let string = String.create n in
    for i = 0 to n - 1 do
      string.[i] <- (read_char s)
    done;
    string

let skip_bytes s n =
  for i = 1 to n do
    ignore(read_byte s)
  done

let read_bytes s n =
  let a = (Array.make n 0) in
    for i = 0 to n - 1 do
      a.(i) <- (read_byte s)
    done;
    a


(* Utilities used by procedures for both formats. *)

let read_style s =
  let style_type = read_byte s in
    if style_type == 8 || style_type == 17 then
      { style_type = style_type; meter = 3 }
    else
      { style_type = style_type; meter = 4 }

let read_key s =
  let key = read_byte s in
    if key <= 17 then
      root_names.(key - 1)
    else
      root_names.(key - 18) ^ "m"

let read_tempo s =
  let tempo1 = read_byte s in
  let tempo2 = read_byte s in
    tempo2 * 256 + tempo1

let chord_name t r = 
  let chord_root = r mod 18 in
  let chord_bass = (chord_root - 1 + r / 18) mod 12 + 1 in
  let root_name = root_names.(chord_root - 1)
  and type_name = Biabchords.to_toe t
  in
    if chord_root = chord_bass then
      root_name ^ type_name
    else
      root_name ^ type_name ^ "/" ^ root_names.(chord_bass - 1)

(* Drop one beat from the last chord of a bar so the total duration is
   three beats. *)
let adjust_for_meter chords meter =
  let make_3_beat bar =
    match List.rev bar with
	(chord, 1) :: tl -> List.rev tl
      | (chord, dur) :: tl -> List.rev ((chord, dur - 1) :: tl)
      | [] -> failwith "Impossible empty bar in adjust_for_meter"
  in
    if meter = 3 then
      List.map make_3_beat chords
    else
      chords


(* Old BiaB file format *)

let read_old_format_name s =
  let name_len = read_byte s in
  let name = read_string s name_len in
    skip_bytes s (0x3e - 2 - name_len);
    trim_right_string name

(* Section information in old format is encoded as 64 bytes (i.e., a
   song contains a maximum of 64 bars).  Each non-zero entry denotes the
   start of a new section (with 1 = "A" and 2 = "B"). *)
let read_sections_old_format s =
  let sections = read_bytes s 64 in
  let result = ref [] in
  let f i section =
    if section <> 0 then
      result := ((i + 1), section_names.(section)) :: !result
  in
    Array.iteri f sections;
    List.rev !result

let read_chorus_old_format s =
  let b = read_byte s in
  let e = read_byte s in
  let r = read_byte s in
    { chorus_begin = b; chorus_end = e; chorus_repeats = r }

(* Chords in old format are encoded as two vectors of 256 bytes each
   representing chord types and chord roots.  A song has a maximum of 64
   bars and thus 256 chords.  In 3/4-time songs, every fourth entry of
   the vectors is unused.  A chord type of zero represents that the
   current beat should be added to the duration of the chord represented
   by the most recent preceding entry with a non-zero chord type. *)
let last_chord = ref ""

let make_bar_old_format types roots =
  let result = ref [] in
    for i = 0 to 3 do
      let chord_type = types.(i)
      and chord_root = roots.(i) in
        if chord_type = 0 then
	  match !result with
	      (c, d) :: tl -> result := (c, d + 1) :: tl
	    | [] -> result := [(!last_chord, 1)]
        else
	  begin
            last_chord := (chord_name chord_type chord_root);
	    result := (!last_chord, 1) :: !result
	  end
    done;
    List.rev !result

let make_bars_old_format types roots length =
  let l = length / 4
  and result = ref [] in
    for bar = 0 to l - 1 do
      let i = bar * 4 in
      let bartypes = (Array.sub types i 4)
      and barroots = (Array.sub roots i 4) in
        result := (make_bar_old_format bartypes barroots) :: !result
    done;
    List.rev !result

let read_old_format_chart s meter =
  let sections = read_sections_old_format s in
  let types = read_bytes s 256 in
  let roots = read_bytes s 256 in
  let _ = read_byte s in
  let chorus = read_chorus_old_format s in
  let bars = make_bars_old_format types roots (chorus.chorus_end * 4) in
    sections, (adjust_for_meter bars meter), chorus

let read_biab_file_old_format s =
  let name = read_old_format_name s in
  let style = read_style s in
  let key = read_key s in
  let tempo = read_tempo s in
  let sections, chords, chorus = read_old_format_chart s style.meter in
    { name = name; style = style; key = key; tempo = tempo; chords = chords; chorus = chorus; sections = sections }


(* New BiaB file format *)

let read_new_format_name s =
  let name_len = read_byte s in
  let name = read_string s name_len in
    skip_bytes s 2;
    trim_right_string name

(* Section information in the new format is variable length because it
   uses a kind of "run-length encoding".  A non-zero section type may be
   followed by a type value of zero followed by a duration (in number of
   bars).  The maximum number of bars in a song encoded in the new format
   is 255. *)
let read_sections_new_format s =
  let rec loop bar result =
    if bar >= 255 then
      List.rev result
    else
      let bar_type = read_byte s in
	if bar_type = 0 then
	  let dur = read_byte s in
	    loop (bar + dur) result
	else if bar_type < Array.length section_names then
	  loop (bar + 1) ((bar, section_names.(bar_type)) :: result)
	else
	  loop (bar + 1) result
  in
    loop (read_byte s) []

(* Chords in the new format are also variable length and are stored in
   a way similar to how section information is stored.  A song has a
   maximum of 255 bars and therefore 4 * 255 = 1020 chords (if all of
   them have duration one).  *)
let read_new_format_chord_types s =
  let rec loop beat result =
    if beat >= 1020 then
      beat, List.rev result
    else
      let chord_type = read_byte s in
	if chord_type = 0 then
	  let dur = read_byte s in
	    match result with
		(chord_type1, dur1) :: tl ->
		  loop (beat + dur) ((chord_type1, dur1 + dur) :: tl)
	      | [] ->
		  failwith "First chord type cannot be 0"
	else
	  loop (beat + 1) ((chord_type, 1) :: result)
  in
    loop 0 []

let read_new_format_chord_roots = read_new_format_chord_types

(* Some new-format BiaB files contain 1021 beats in the chord roots
   vector and some 1020 beats.  In the former, the chorus information
   follows immediately and in the latter, one byte needs to be skipped to
   get to the chorus information.  That's why the procedure
   read-new-format-chord-roots must also return the total beats in the
   chord roots vector and this number must be passed to
   read-chorus-new-format. *)
let read_chorus_new_format s beat =
  if beat = 1020 then
    skip_bytes s 1;
  let b = read_byte s in
  let e = read_byte s in
  let r = read_byte s in
    { chorus_begin = b; chorus_end = e; chorus_repeats = r + 2 }

(* Fix two problems with BiaB output: chord type and root sequences
   may have different lengths; last duration may not be 4 (or 3), *)
let sanitize n l =
  let rec loop n result rest =
    if n <= 0 then
      List.rev result
    else
      match rest with
	  (x, _) :: [] -> loop (n - 1) ((x, 4) :: result) []
	| hd :: tl -> loop (n - 1) (hd :: result) tl
	| [] -> failwith "List shorter than n"
  in
    loop n [] l

let take_one_bar l =
  let rec loop dur bar l =
    match l with
	[] -> List.rev bar, []
      | (c, d) :: tl ->
	  let new_dur = dur + d in
	    if new_dur < 4 then
	      loop new_dur ((c,d) :: bar) tl
	    else if new_dur = 4 then
	      List.rev ((c, d) :: bar), tl
	    else
	      let part1 = 4 - dur in
	      let part2 = d - part1 in
		List.rev ((c, part1) :: bar), (c, part2) :: tl
  in
    loop 0 [] l

let construct_bars l =
  let rec loop l result =
    if l = [] then
      List.rev result
    else
      let bar, rest = take_one_bar l in
	loop rest (bar :: result)
  in
    loop l []

(* Turn lists of elements of the forms (chord_type, duration) and
   (chord_root, duration) into a list with elements of the form
   (chord-name . duration).  Then pass it on to be processed by
   construct_bars.  *)
let make_bars_new_format types roots =
  let l1 = List.length types and l2 = List.length roots in
    (* Sometimes they're off by 1!  It's probably a BiAB output code
       bug.  *)
    if abs (l1 - l2) <= 1 then
      let len = min l1 l2
      and f (t, d1) (r, d2) = (*assert (d1 = d2);*) (chord_name t r), d1
      in
	construct_bars (List.map2 f (sanitize len types) (sanitize len roots))
    else
      failwith "lengths of roots and types differ by more than 1"

let read_new_format_chart s meter =
  let sections = read_sections_new_format s in
  let _, types = read_new_format_chord_types s in
  let beat, roots = read_new_format_chord_roots s in
  let chorus = read_chorus_new_format s beat in
  let bars = make_bars_new_format types roots in
    sections, (adjust_for_meter bars meter), chorus

let read_biab_file_new_format s =
  let name = read_new_format_name s in
  let style = read_style s in
  let key = read_key s in
  let tempo = read_tempo s in
  let sections, chords, chorus = read_new_format_chart s style.meter in
    { name = name; style = style; key = key; tempo = tempo; chords = chords; chorus = chorus; sections = sections }


(* "main" *)

let read_chart_stream s =
  let version = read_byte s in
    if version = 0xbb || version = 0xbc then
      read_biab_file_old_format s
    else
      read_biab_file_new_format s

let read_chart s =
  read_chart_stream (Stream.of_string s)

let read fname =
  let ic = open_in_bin fname in
  let chart = read_chart_stream (Stream.of_channel ic) in
    close_in ic;
    chart

(*let test () =
  read "testfiles/allofyou.mgu";;

let readfile fname =
  let ic = open_in_bin fname in
    let rec loop s =
      try
	let c = input_char ic in
	  loop (s ^ (String.make 1 c))
      with
	_ ->
	  s
    in
      loop ""*)
