(*

The main purpose of the release of this code is to demonstrate the
author's T2 algorithm for harmonic analysis of jazz chord sequences.

Permission for the use of this code is granted only for research,
educational, and non-commercial purposes.

Redistribution of this code or its parts in source, binary,
and any other form without permission, with or without modification,
is prohibited.  Modifications include, but are not limited to,
translation to other programming languages and reuse of tables,
constant definitions, and API's defined in it.

Andrew Choi is not liable for any losses or damages caused by the use
of this software.

Copyright 2008 Andrew Choi.
http://www.sixthhappiness.ca/T2/index.html

*)

open Utils
open Chart

let key_re =
  let k = "\\(C#\\|C\\|D#\\|Db\\|D\\|Eb\\|E\\|F#\\|F\\|G#\\|Gb\\|G\\|A#\\|Ab\\|A\\|Bb\\|B\\)\\(m?\\)" in
    Str.regexp ("^" ^ k ^ "$")

let root_names = [|"C"; "Db"; "D"; "Eb"; "E"; "F"; "Gb"; "G"; "Ab"; "A"; "Bb"; "B"; "C#"; "D#"; "F#"; "G#"; "A#"|]

let chord_re =
  let n = "\\([A-G]\\)" and alt = "\\(\\(##\\|#\\|bb\\|b\\)?\\)" in
    let note = "\\(" ^ n ^ alt ^ "\\)" in
      Str.regexp ("^" ^ note ^ "\\([^/]*\\)\\(/" ^ note ^ "\\)?$")

let write_char buf c =
  Buffer.add_char buf c

let write_integer_as_char buf i =
  assert (i < 256);
  write_char buf (char_of_int i)

let write_string buf s =
  let len = String.length s in
    for i = 0 to len - 1 do
      write_char buf s.[i]
    done

let write_filler1 buf =
  write_char buf '\x44'

let write_name buf name =
  let l = String.length name in
    write_integer_as_char buf l;
    write_string buf name;
    (* Write "run-length encoded" zero padding to a length of 0x4c. *)
    write_integer_as_char buf 0;
    write_integer_as_char buf (0x3c - l)

let write_style buf style_type =
  write_integer_as_char buf style_type

let write_key buf key =
  if Str.string_match key_re key 0 then
    let r = Str.matched_group 1 key in
      if Str.matched_group 2 key = "m" then
	write_integer_as_char buf ((index_array root_names r) + 18)
      else
	write_integer_as_char buf ((index_array root_names r) + 1)
  else
    failwith ("Invalid key: " ^ key)

let write_tempo buf tempo =
  let t1 = tempo mod 256
  and t2 = tempo / 256
  in
    write_integer_as_char buf t1;
    write_integer_as_char buf t2

let write_filler2 buf =
  (* This seems to be the starting bar number (always using 1 serves
  our purpose). *)
  write_char buf '\x01'

(* Sort sections by bar index and make sure bar 1 has a section mark. *)
let cleanup_sections sections =
  let s = List.sort (fun (i, _) (j, _) -> Pervasives.compare i j) sections in
    match s with
      (1, _) :: _ -> s
    | l -> (1, 'A') :: l

let write_sections buf sections =
  let lastprev =
    List.fold_left
      (fun prev (b, t) ->
	if prev > 0 then
	  begin
	    write_integer_as_char buf 0;
	    write_integer_as_char buf (b - prev - 1)
	  end;
	
	begin
	  match t with
	    'A' -> write_integer_as_char buf 1
	  | 'B' -> write_integer_as_char buf 2
	  | _ -> failwith ("Invalid section type: " ^ (String.make 1 t))
	end;
	
	b)
      0
      sections
    in
      write_integer_as_char buf 0;
      write_integer_as_char buf (255 - lastprev)

let rec remove_beat_from_last_chord bar =
  match bar with
    (c, d) :: [] -> (c, d - 1) :: []
  | p :: tl -> p :: (remove_beat_from_last_chord tl)
  | _ -> failwith "Illegal empty bar in adjust_bar_to_4_beats"

let adjust_for_meter chords meter =
  match meter with
    4 -> chords
  | 3 -> List.map remove_beat_from_last_chord chords
  | _ -> failwith ("Unrecognized meter: " ^ (string_of_int meter))

let extract_types_and_roots chords =
  List.map
    (fun c ->
      if Str.string_match chord_re c 0 then
	try
	  let r = (index_array root_names (Str.matched_group 1 c)) + 1
	  and t = Biabchords.of_toe (Str.matched_group 5 c)
	  in
	    let r2 =
	      try
		let b = (index_array root_names (Str.matched_group 7 c)) + 1 in
		  r + ((b + 12 - r) mod 12) * 18
	      with Not_found ->
		r
	    in
	      (t, r2)
	with Not_found ->
	  failwith ("Cannot convert chord " ^ c ^ " to BiaB")
      else
	failwith ("Unrecognized chord: " ^ c))
    chords

let write_chord_types buf l =
  let beat = ref 0 in
    List.iter
      (fun (t, d) ->
	write_integer_as_char buf t;
	if d > 1 then
	  begin
	    write_integer_as_char buf 0;
	    write_integer_as_char buf (d - 1)
	  end;
	beat := !beat + d)
      l;

    while !beat <= 1020 - 255 do
      write_integer_as_char buf 0;
      write_integer_as_char buf 255;
      beat := !beat + 255
    done;

    if !beat < 1020 then
      begin
	write_integer_as_char buf 0;
	write_integer_as_char buf (1020 - !beat)
      end

let write_chord_roots = write_chord_types

let write_chords buf chords style =
  let adjusted_chords = adjust_for_meter chords style.meter in
    let chords, durations = List.split (List.concat adjusted_chords) in
      let types, roots = List.split (extract_types_and_roots chords) in
	write_chord_types buf (List.combine types durations);
	write_chord_roots buf (List.combine roots durations)

let write_filler3 buf =
  (* Extra byte that needs to appear after the 1020 beat chord root vector. *)
  write_char buf '\x01'

let write_chorus buf chorus =
  write_integer_as_char buf chorus.chorus_begin;
  write_integer_as_char buf chorus.chorus_end;
  write_integer_as_char buf chorus.chorus_repeats

let write_filler4 buf chorus_end =
  write_string buf "\x01\x00\x01";

  write_integer_as_char buf chorus_end;
  write_integer_as_char buf (chorus_end + 1);
  write_integer_as_char buf (chorus_end + 4);

  write_string buf "\x0b\x0b\x0b\x0b";

  for i = 1 to 6 do
    write_string buf "\x00\xff"
  done;

  write_string buf "\x00\x06";

  for i = 1 to 65 do
    write_string buf "\x01\x20\x00\x4d"
  done;

  write_string buf "\x42";

  for i = 1 to 9 do
    write_string buf "\x00\xff"
  done;

  write_string buf "\x00\x2e\x01\x01\x01\x00\x04\x01\xe1\x00\xc8\xa0\xb0\xc0"

let write_file_length buf =
  let pos = Buffer.length buf in
    write_integer_as_char buf (pos land 0xff);
    write_integer_as_char buf ((pos lsr 8) land 0xff);
    write_integer_as_char buf ((pos lsr 16) land 0xff);
    write_integer_as_char buf ((pos lsr 24) land 0xff)

let write_chart chart =
  let buf = Buffer.create 16 in

    write_filler1 buf;

    write_name buf chart.name;
    write_style buf chart.style.style_type;
    write_key buf chart.key;
    write_tempo buf chart.tempo;

    write_filler2 buf;

    write_sections buf (cleanup_sections chart.sections);
    write_chords buf chart.chords chart.style;

    write_filler3 buf;

    write_chorus buf chart.chorus;

    write_filler4 buf chart.chorus.chorus_end;
    write_file_length buf;

    Buffer.contents buf

let test () =
  let chart =
    {name = "ALL OF YOU"; style = {meter = 4; style_type = 1}; key = "Eb";
     tempo = 165;
     chords =
     [[("Bb7b9", 4)]; [("EbMaj7", 4)]; [("Bb7b9", 4)]; [("Bb7b9", 4)];
      [("Bb7b9", 4)]; [("EbMaj7", 4)]; [("Fm7b5", 4)]; [("Bb9", 4)];
      [("Eb6", 4)]; [("Gb:b3.b5", 4)]; [("Fm7", 4)]; [("Bb9", 4)];
      [("EbMaj7", 2); ("Eb/D", 2)]; [("Gm7b5/Db", 2); ("C7", 2)]; [("Fm7", 4)];
      [("Bb9", 4)]; [("Bb7b9", 4)]; [("EbMaj7", 4)]; [("Bb7b9", 4)];
      [("Bb7b9", 4)]; [("Bb7b9", 4)]; [("EbMaj7", 4)]; [("Gm7", 4)];
      [("C7", 4)]; [("AbMaj7", 4)]; [("Am7b5", 2); ("D7b9", 2)]; [("Gm7", 4)];
      [("C:4.5.b7.b9", 2); ("C7", 2)]; [("Fm7", 2); ("C7", 2)];
      [("Fm7", 2); ("Bb7", 2)]; [("Eb6", 4)]; [("Eb6", 4)]; [("Eb69", 4)]];
     chorus = {chorus_begin = 1; chorus_end = 32; chorus_repeats = 3};
     sections = [(1, 'A'); (17, 'A'); (33, 'A')]}
  in
    let oc = open_out_bin "testfiles/biabwriter.sg1" in
      output_string oc (write_chart chart);
      close_out oc
