(*

The main purpose of the release of this code is to demonstrate the
author's T2 algorithm for harmonic analysis of jazz chord sequences.

Permission for the use of this code is granted only for research,
educational, and non-commercial purposes.

Redistribution of this code or its parts in source, binary,
and any other form without permission, with or without modification,
is prohibited.  Modifications include, but are not limited to,
translation to other programming languages and reuse of tables,
constant definitions, and API's defined in it.

Andrew Choi is not liable for any losses or damages caused by the use
of this software.

Copyright 2008 Andrew Choi.
http://www.sixthhappiness.ca/T2/index.html

*)

type t = { 
  root: Note.t;
  chordtype: Chordtype.t;
  bass: Note.t;
  notes: Note.t list }

let chord_re =
  let n = "\\([A-G]\\)" and alt = "\\(\\(##\\|#\\|bb\\|b\\)?\\)" in
  let note = "\\(" ^ n ^ alt ^ "\\)" in
    Str.regexp ("^" ^ note ^ "\\([^/]*\\)\\(/" ^ note ^ "\\)?$")

let of_string s =
  let notes root chordtype =
    List.map (fun intvl -> (Note.add_intvl root intvl)) chordtype in
    if Str.string_match chord_re s 0 then
      let r = Str.matched_group 1 s and c = Str.matched_group 5 s in
      let b = try Str.matched_group 7 s with Not_found -> "" in
	try
	  let rr = Note.of_string r and cc = Chordtype.of_string c in
	    if b <> "" then
	      let bb = Note.of_string b in
		{ root = rr; chordtype = cc; bass = bb; notes = (notes rr cc) }
	    else
	      { root = rr; chordtype = cc; bass = rr; notes = (notes rr cc) }
	with Not_found ->
	  failwith ("Unknown chord type: " ^ c)
    else
      failwith ("Invalid chord: " ^ s)

let to_string chord =
  let c = Note.to_string chord.root ^ Chordtype.to_string chord.chordtype in
    if chord.bass = chord.root then
      c
    else
      c ^ "/" ^ (Note.to_string chord.bass)

let pp formatter x = Format.pp_print_string formatter (to_string x)

let nth chord n =
  Note.add_intvl chord.root (Chordtype.nth chord.chordtype n)

(* Getters *)
let root chord = chord.root
let chordtype chord = chord.chordtype
let bass chord = chord.bass
let notes chord = chord.root :: chord.notes

let compare c1 c2 =
  let r = Note.compare c1.root c2.root in
    if r = 0 then
      Intvllist.compare c1.chordtype c2.chordtype
    else
      r

include Chorddefs
