(*

The main purpose of the release of this code is to demonstrate the
author's T2 algorithm for harmonic analysis of jazz chord sequences.

Permission for the use of this code is granted only for research,
educational, and non-commercial purposes.

Redistribution of this code or its parts in source, binary,
and any other form without permission, with or without modification,
is prohibited.  Modifications include, but are not limited to,
translation to other programming languages and reuse of tables,
constant definitions, and API's defined in it.

Andrew Choi is not liable for any losses or damages caused by the use
of this software.

Copyright 2008 Andrew Choi.
http://www.sixthhappiness.ca/T2/index.html

*)

open Constants

type t = { note: Note.t; octave: int }

let midinote_re =
  let n = "\\([A-G]\\)" and alt = "\\(\\(##\\|#\\|bb\\|b\\)?\\)" in
  let note = "\\(" ^ n ^ alt ^ "\\)" in
    Str.regexp ("^" ^ note ^ "\\(-1\\|[0-9]\\)" ^ "$")

let deflt_notes = Array.map Note.of_string [| "C"; "Db"; "D"; "Eb"; "E"; "F"; "F#"; "G"; "Ab"; "A"; "Bb"; "B" |]

let to_notenum m =
  let n, alt = Note.to_n_alt m.note in
    (m.octave + 1) * 12 + semis.(n) + alt

let validate m =
  let n = to_notenum m in
    if n < 0 || n > 127 then
      raise (Invalid_argument ("Midinote out of range: " ^ (string_of_int n)))
    else
      m

let of_notenum n =
  let m = { note = deflt_notes.(n mod 12); octave = n / 12 - 1 } in
    validate m

let of_note_octave note octave =
  let m = { note = note; octave = octave } in
    validate m

let to_note_octave m =
  m.note, m.octave

let of_string s =
  if Str.string_match midinote_re s 0 then
    let note = Str.matched_group 1 s and octave = Str.matched_group 5 s in
    let m = { note = Note.of_string note; octave = int_of_string octave } in
      validate m
  else
    failwith ("Invalid midinote: " ^ s)

let to_string m =
  (Note.to_string m.note) ^ (string_of_int m.octave)

let pp formatter x = Format.pp_print_string formatter (to_string x)

(* Getters *)
let note m = m.note
let octave m = m.octave

let compare m1 m2 =
  Pervasives.compare (m1.note, m1.octave) (m2.note, m2.octave)

let equal m1 m2 =
  (m1.note, m1.octave) = (m2.note, m2.octave)

let hash m =
  Hashtbl.hash (m.note, m.octave)

let add_int m i =
  of_notenum (to_notenum m + i)

let sub_int m i =
  of_notenum (to_notenum m - i)

let sub_midinote m1 m2 =
  to_notenum m1 - to_notenum m2

let of_notenum_in_notelist notenum notelist =
  let note = List.find (fun n -> Note.semi n = notenum mod 12) notelist in
    let n, alt = Note.to_n_alt note in
      (* semi can be -2, -1, ..., 12, 13 for Cbb, Cb, ..., B#, B##.  *)
      let semi = semis.(n) + alt in
	{ note = note; octave = (notenum - semi) / 12 - 1 }
