(*

The main purpose of the release of this code is to demonstrate the
author's T2 algorithm for harmonic analysis of jazz chord sequences.

Permission for the use of this code is granted only for research,
educational, and non-commercial purposes.

Redistribution of this code or its parts in source, binary,
and any other form without permission, with or without modification,
is prohibited.  Modifications include, but are not limited to,
translation to other programming languages and reuse of tables,
constant definitions, and API's defined in it.

Andrew Choi is not liable for any losses or damages caused by the use
of this software.

Copyright 2008 Andrew Choi.
http://www.sixthhappiness.ca/T2/index.html

*)

open Num

let key_re =
  let key = "\\([A-G]\\|F#\\|C#\\|Bb\\|Eb\\|Ab\\|Db\\|Gb\\|Cb\\)" in
    Str.regexp ("^" ^ key ^ "\\(m\\)?$")

let sharp_flat_codes = [("C", 0); ("G", 1); ("D", 2); ("A", 3); ("E", 4); ("B", 5); ("F#", 6); ("C#", 7); ("F", -1); ("Bb", -2); ("Eb", -3); ("Ab", -4); ("Db", -5); ("Gb", -6); ("Cb", -7)]

let ticks_per_quarter_note = 480
let midi_clock_per_click = 24
let thirty_second_notes_per_quarter_note = 8

let write_byte buf i =
  Buffer.add_char buf (char_of_int i)

let write_32 buf i =
  write_byte buf (i lsr 24 land 0xff);
  write_byte buf (i lsr 16 land 0xff);
  write_byte buf (i lsr 8 land 0xff);
  write_byte buf (i land 0xff)

let write_24 buf i =
  write_byte buf (i lsr 16 land 0xff);
  write_byte buf (i lsr 8 land 0xff);
  write_byte buf (i land 0xff)

let write_16 buf i =
  write_byte buf (i lsr 8 land 0xff);
  write_byte buf (i land 0xff)

let write_8 buf i =
  write_byte buf (i land 0xff)

let write_string buf s =
  String.iter (fun c -> write_byte buf (int_of_char c)) s

let write_delta_time buf i =
  if i > 0xfffffff then
    failwith ("Value exceeds 0xfffffff: " ^ (string_of_int i))
  else
    let int_to_string i = String.make 1 (char_of_int i) in
    let rec loop s i =
      if i > 0 then
	loop ((int_to_string ((i land 0x7f) lor 0x80)) ^ s) (i lsr 7)
      else
	s
    in
      write_string buf (loop (int_to_string (i land 0x7f)) (i lsr 7))

let write_chunk_type buf ct =
  assert (String.length ct = 4);
  write_string buf ct

let write_header_chunk buf ntrks =
  write_chunk_type buf "MThd";
  write_32 buf 6;  (* length of header data *)
  write_16 buf 1;  (* format *)
  write_16 buf ntrks;
  write_16 buf ticks_per_quarter_note  (* division *)

let write_sequence_or_track_name buf name =
  let l = String.length name in
    if l > 256 then
      failwith ("Sequence/track name exceeds 256 chars: " ^ name)
    else
      begin
	write_8 buf 0xff;  (* meta *)
	write_8 buf 0x03;  (* sequence or track name *)
	write_8 buf l;  (* length *)
	
	write_string buf name
      end

let write_time_signature buf num den =
  let rec log2 i = if i > 1 then (log2 (i lsr 1)) + 1 else 0 in
    write_8 buf 0xff;  (* meta *)
    write_8 buf 0x58;  (* time signature *)
    write_8 buf 0x04;  (* length *)
    
    write_8 buf num;
    write_8 buf (log2 den);
    write_8 buf midi_clock_per_click;
    write_8 buf thirty_second_notes_per_quarter_note

let write_set_tempo buf bpm =
  write_8 buf 0xff;  (* meta *)
  write_8 buf 0x51;  (* set tempo *)
  write_8 buf 0x03;  (* length *)
    
  write_24 buf (60000000 / bpm)

let write_key_signature buf key =
  if Str.string_match key_re key 0 then
    let k = Str.matched_group 1 key in
      begin
	write_8 buf 0xff;  (* meta *)
	write_8 buf 0x59;  (* set tempo *)
	write_8 buf 0x02;  (* length *)
	
	write_8 buf (List.assoc k sharp_flat_codes)
      end;

      try
	let _ = Str.matched_group 2 key in
	  write_8 buf 1  (* minor *)
      with _ ->
	write_8 buf 0  (* major *)
  else
    failwith ("Invalid key: " ^ key)

let write_end_of_track buf =
  write_8 buf 0xff;  (* meta *)
  write_8 buf 0x2f;  (* end of track *)
  write_8 buf 0x00  (* length *)

let write_tempo_track_chunk buf name meter tempo key =
  write_chunk_type buf "MTrk";

  let buf2 = Buffer.create 16 in
    write_delta_time buf2 0;
    write_sequence_or_track_name buf2 name;
    
    write_delta_time buf2 0;
    write_time_signature buf2 (fst meter) (snd meter);
    
    write_delta_time buf2 0;
    write_set_tempo buf2 tempo;
    
    write_delta_time buf2 0;
    write_key_signature buf2 key;
    
    write_delta_time buf2 ticks_per_quarter_note;
    write_end_of_track buf2;

    write_32 buf (Buffer.length buf2);
    Buffer.add_buffer buf buf2

let write_program_change buf channel pc =
  assert (channel >= 1 && channel <= 16);
  assert (pc >= 1 && pc <= 256);

  write_8 buf (0xc0 lor (channel - 1));
  write_8 buf (pc - 1)

let write_note_on buf channel pitch velocity =
  assert (channel >= 1 && channel <= 16);
  assert (pitch >= 0 && pitch <= 127);
  assert (velocity >= 0 && velocity <= 127);

  write_8 buf (0x90 lor (channel - 1));
  write_8 buf pitch;
  write_8 buf velocity

let write_note_off buf channel pitch =
  assert (channel >= 1 && channel <= 16);
  assert (pitch >= 0 && pitch <= 127);

  write_8 buf (0x80 lor (channel - 1));
  write_8 buf pitch;
  write_8 buf 64  (* standard velocity for note off *)

type noteonoff =
      Noteon of int * int
    | Noteoff of int

let write_sequence buf seq channel =
  let midievents = ref [] in
  begin
    let add event =
      let (ts, dur, (m, vel)) = event in
	let nn = (Midinote.to_notenum m) in
	  midievents := (ts, (Noteon (nn, vel))) :: !midievents;
	  midievents := ((ts +/ dur), (Noteoff nn)) :: !midievents
    in
      List.iter add (Seq.events seq);

      let curr_time = ref (Int 0) in
	let write (ts, noteonoff) =
	  let ticks = (Int ticks_per_quarter_note) in
	    write_delta_time buf (int_of_num ((ts -/ !curr_time) */ ticks));
	    curr_time := ts;
	    match noteonoff with
	      Noteon (notenum, vel) ->
		write_note_on buf channel notenum vel
	    | Noteoff notenum ->
		write_note_off buf channel notenum in

	  (* Noteoff's must be sorted before Noteon's for consecutive
	  sequences of the same MIDI notes to work correctly.  *)

	  let compare_midievents (ts1, e1) (ts2, e2) =
	    let r = compare_num ts1 ts2 in
	      if r <> 0 then
		r
	      else
		match e1, e2 with
		  Noteoff _, Noteon (_, _) -> -1
		| Noteon (_, _), Noteoff _ -> 1
		| _ -> 0
	  in
	    List.iter write (List.sort compare_midievents !midievents)
  end

let write_instrument_track_chunk buf seqinfo =
  let seq, name, channel, pc = seqinfo in
    write_chunk_type buf "MTrk";

    let buf2 = Buffer.create 16 in
      write_delta_time buf2 0;
      write_sequence_or_track_name buf2 name;
      
      write_delta_time buf2 0;
      write_program_change buf2 channel pc;

      write_sequence buf2 seq channel;

      (* Allow decay of last note to finish.  *)
      write_delta_time buf2 ticks_per_quarter_note;
      write_end_of_track buf2;

      write_32 buf (Buffer.length buf2);
      Buffer.add_buffer buf buf2

let write seqs name meter tempo key =
  let buf = Buffer.create 16 in
    write_header_chunk buf ((List.length seqs) + 1);
    write_tempo_track_chunk buf name meter tempo key;
    List.iter (write_instrument_track_chunk buf) seqs;
    Buffer.contents buf

let test () =
  let oc = open_out_bin "testfiles/testfile.mid" in
    let add_event x s =
      let ts, note, dur = x in
	Seq.add (Int ts) (Int dur) ((Midinote.of_string note), 64) s
    in
      output_string oc (write [((List.fold_right add_event [(1, "C4", 1); (2, "D4", 1); (3, "E4", 1); (4, "F#4", 1)] (Seq.create ())), "piano", 1, 1)] "test" (4, 4) 120 "C");
      close_out oc
