(*

The main purpose of the release of this code is to demonstrate the
author's T2 algorithm for harmonic analysis of jazz chord sequences.

Permission for the use of this code is granted only for research,
educational, and non-commercial purposes.

Redistribution of this code or its parts in source, binary,
and any other form without permission, with or without modification,
is prohibited.  Modifications include, but are not limited to,
translation to other programming languages and reuse of tables,
constant definitions, and API's defined in it.

Andrew Choi is not liable for any losses or damages caused by the use
of this software.

Copyright 2008 Andrew Choi.
http://www.sixthhappiness.ca/T2/index.html

*)

open Constants
open Utils

type t = { n: int; alt: int }

let note_re =
  let n = "\\([A-G]\\)" and alt = "\\(\\(##\\|#\\|bb\\|b\\)?\\)" in
    Str.regexp ("^" ^ n ^ alt ^ "$")

let of_string s =
  if Str.string_match note_re s 0 then
    let n = Str.matched_group 1 s and alt = Str.matched_group 2 s in
      {n = index_array note_names n; alt = (index_array alt_names alt) - 2}
  else
    failwith ("Invalid note: " ^ s)

let to_string note =
  note_names.(note.n) ^ alt_names.(note.alt + 2)

let pp formatter x = Format.pp_print_string formatter (to_string x)

let of_n_alt n alt =
  if n < 0 || n > 6 then
    failwith ("Invalid n: " ^ (string_of_int n))
  else if alt < -2 || alt > 2 then
    failwith ("Invalid alt: " ^ (string_of_int alt))
  else
    { n = n; alt = alt }

let to_n_alt note =
  note.n, note.alt

let compare n1 n2 =
  Pervasives.compare (n1.n, n1.alt) (n2.n, n2.alt)

let equal n1 n2 =
  (n1.n, n1.alt) = (n2.n, n2.alt)

let hash note =
  Hashtbl.hash (note.n, note.alt)

let semi note =
  (semis.(note.n) + note.alt + 12) mod 12

let add_intvl n i =
  let i_i, i_alt = Intvl.to_i_alt i in
  let res_n = (n.n + i_i) mod 7 in
  let note_semi_incr = (semis.(res_n) - semis.(n.n) + 12) mod 12 in
  let res_semi_incr = (semis.(i_i) + i_alt + 12) mod 12 in
  let res_alt = (n.alt + res_semi_incr - note_semi_incr + 14) mod 12 - 2 in
    if res_alt < -2 || res_alt > 2 then
      failwith ((to_string n) ^ " + " ^ (Intvl.to_string i) ^ " results in invalid note")
    else
      {n = res_n; alt = res_alt}

let sub_intvl n i =
  let i_i, i_alt = Intvl.to_i_alt i in
  let res_n = (n.n - i_i + 7) mod 7 in
  let note_semi_decr = (semis.(n.n) - semis.(res_n) + 12) mod 12 in
  let res_semi_decr = (semis.(i_i) + i_alt + 12) mod 12 in
  let res_alt = (n.alt + note_semi_decr - res_semi_decr + 14) mod 12 - 2 in
    if res_alt < -2 || res_alt > 2 then
      failwith ((to_string n) ^ " - " ^ (Intvl.to_string i) ^ " results in invalid note")
    else
      {n = res_n; alt = res_alt}

let sub_note n1 n2 =
  let res_i = (n1.n - n2.n + 7) mod 7 in
  let note_semi_diff = (semis.(n1.n) + n1.alt - semis.(n2.n) - n2.alt + 12) mod 12 in
  let res_alt = (note_semi_diff - semis.(res_i) + 14) mod 12 - 2 in
    if res_alt < -2 || res_alt > 2 then
      failwith ((to_string n1) ^ " - " ^ (to_string n2) ^ " results in invalid interval")
    else
      Intvl.of_i_alt res_i res_alt
