(*

The main purpose of the release of this code is to demonstrate the
author's T2 algorithm for harmonic analysis of jazz chord sequences.

Permission for the use of this code is granted only for research,
educational, and non-commercial purposes.

Redistribution of this code or its parts in source, binary,
and any other form without permission, with or without modification,
is prohibited.  Modifications include, but are not limited to,
translation to other programming languages and reuse of tables,
constant definitions, and API's defined in it.

Andrew Choi is not liable for any losses or damages caused by the use
of this software.

Copyright 2008 Andrew Choi.
http://www.sixthhappiness.ca/T2/index.html

*)

type t = { 
  root: Note.t;
  scaletype: Scaletype.t;
  notes: Note.t list }

let scale_re =
  let n = "\\([A-G]\\)" and alt = "\\(\\(##\\|#\\|bb\\|b\\)?\\)" in
  let note = "\\(" ^ n ^ alt ^ "\\)" in
    Str.regexp ("^" ^ note ^ "\\([ \\t]*\\)\\(.+\\)$")

let of_string n =
  let notes root scaletype =
    List.map (fun intvl -> (Note.add_intvl root intvl)) scaletype in
    if Str.string_match scale_re n 0 then
      let r = Str.matched_group 1 n and s = Str.matched_group 6 n in
	try
	  let rr = Note.of_string r and ss = Scaletype.of_string s in
	    { root = rr; scaletype = ss; notes = (notes rr ss) }
	with Not_found ->
	  failwith ("Unknown scale type: " ^ s)
    else
      failwith ("Invalid scale: " ^ n)

let to_string scale =
  let scaletypename = Scaletype.to_string scale.scaletype in
    if String.length scaletypename > 0 && scaletypename.[0] = ':' then
      Note.to_string scale.root ^ scaletypename
    else
      Note.to_string scale.root ^ " " ^ scaletypename

let pp formatter x = Format.pp_print_string formatter (to_string x)

let nth scale n =
  Note.add_intvl scale.root (Scaletype.nth scale.scaletype n)

(* Getters *)
let root scale = scale.root
let scaletype scale = scale.scaletype
let notes scale = scale.root :: scale.notes

let compare s1 s2 =
  let r = Note.compare s1.root s2.root in
    if r = 0 then
      Intvllist.compare s1.scaletype s2.scaletype
    else
      r

include Scaledefs
