(*

The main purpose of the release of this code is to demonstrate the
author's T2 algorithm for harmonic analysis of jazz chord sequences.

Permission for the use of this code is granted only for research,
educational, and non-commercial purposes.

Redistribution of this code or its parts in source, binary,
and any other form without permission, with or without modification,
is prohibited.  Modifications include, but are not limited to,
translation to other programming languages and reuse of tables,
constant definitions, and API's defined in it.

Andrew Choi is not liable for any losses or damages caused by the use
of this software.

Copyright 2008 Andrew Choi.
http://www.sixthhappiness.ca/T2/index.html

*)

open Chart
open Utils

let blank_re = Str.regexp ("^[ \t\n]*$")

let headerline_re =
  let kw = "\\(Name\\|Meter\\|Style\\|Key\\|Tempo\\|Chorus\\|Sections\\)" in
    Str.regexp ("^" ^ kw ^ ":[ \t]*\\(.*\\)$")

let chorus_re =
  let num = "\\([0-9]+\\)" in
    Str.regexp ("^bars " ^ num ^ " to " ^ num ^ " (repeat " ^ num ^ " times)$")

let section_re = Str.regexp "^\\([0-9]+\\)\\(A\\|B\\)$"

let input_line_stream s =
  let rec loop l =
    let c = Stream.next s in
      if c = '\n' then
	l
      else
	loop (l ^ (String.make 1 c))
  in
    loop ""

let parse_bar bar meter prevchord =
  let chords = Str.split (Str.regexp "[ \t]+") bar in
  let durunit = meter / (List.length chords) in
  let rec loop result chords =
    match chords with
	[] -> List.rev result
      | hd :: tl ->
	  if hd <> "_" then
	    begin
	      prevchord := hd;
	      loop ((hd, durunit) :: result) tl
	    end
	  else
	    match result with
		(c, d) :: tl2 -> loop ((c, d + durunit) :: tl2) tl
	      | [] -> 
		  if !prevchord = "" then
		    failwith "First chord cannot be _"
		  else
		    loop ((!prevchord, durunit) :: result) tl
  in
    loop [] chords

let parse_chords line meter prevchord =
  let bars = Str.split (Str.regexp "|") line
  and parse_and_append_bar result bar =
    result @ [(parse_bar bar meter prevchord)]
  in
    List.fold_left parse_and_append_bar [] bars

let parse_meter line =
  if line = "3/4" then 3 else 4

let parse_style_type line =
  match line with
      "Jazz Waltz" -> 8
    | "Bossa Nova" -> 4
    | _ -> 1

let parse_chorus line =
  if Str.string_match chorus_re line 0 then
    let m i = int_of_string (Str.matched_group i line) in
      { chorus_begin = m 1; chorus_end = m 2; chorus_repeats = m 3 }
  else
    failwith ("Invalid chorus line: " ^ line)

let parse_sections line =
  let sections = Str.split (Str.regexp "[ \t]+") line
  and parse_section s =
    if Str.string_match section_re s 0 then
      (int_of_string (Str.matched_group 1 s)), (Str.matched_group 2 s).[0]
    else
      failwith ("Invalid section specification: " ^ s)
  in
    List.map parse_section sections

let read_chart_stream s =
  let name = ref ""
  and meter = ref  4
  and style_type = ref 1
  and key = ref "C"
  and tempo = ref 120
  and chorus = ref { chorus_begin = 1; chorus_end = 1; chorus_repeats = 1 }
  and sections = ref []
  and chords = ref []
  and prevchord = ref ""
  in
    begin
      try
	while true do
	  let l = input_line_stream s in
	    if Str.string_match blank_re l 0 then
	      ()
	    else if l.[0] = '|' then
	      chords := !chords @ (parse_chords l !meter prevchord)
	    else if Str.string_match headerline_re l 0 then
	      let k = Str.matched_group 1 l and v = Str.matched_group 2 l in
		match String.lowercase k with
		    "name" -> name := (trim_right_string v)
		  | "meter" -> meter := (parse_meter v)
		  | "style" -> style_type := (parse_style_type v)
		  | "key" -> key := (trim_right_string v)
		  | "tempo" -> tempo := (int_of_string (trim_right_string v))
		  | "chorus" -> chorus := (parse_chorus v)
		  | "sections" -> sections := (parse_sections v)
		  | _ -> failwith ("Unknown header line keyword: " ^ k)
	    else
	      failwith ("Invalid input line: " ^ l)
	done
      with
	  Stream.Failure -> ()
    end;	    
    { name = !name; style = { style_type = !style_type; meter = !meter }; key = !key; tempo = !tempo; chords = !chords; chorus = !chorus; sections = !sections }

let read_chart s =
  read_chart_stream (Stream.of_string s)

let read fname =
  let ic = open_in_bin fname in
  let chart = read_chart_stream (Stream.of_channel ic) in
    close_in ic;
    chart

let test () =
  read "testfiles/Blues #40.mjb";;

(*let readfile fname =
  let ic = open_in_bin fname in
    let rec loop s =
      try
	let c = input_char ic in
	  loop (s ^ (String.make 1 c))
      with
	_ ->
	  s
    in
      loop ""*)
